import { system, world, ItemStack, BlockPermutation, GameMode, EquipmentSlot, ItemEnchantableComponent, Direction, MolangVariableMap } from "@minecraft/server";
import { ModalFormData, ActionFormData } from "@minecraft/server-ui";
import { itemApplyDamage, addItemOrSpawn, handleSitOnFurniture, decrementItemInHand, isCreative, spawnItemSilkTouch, getDirectionByPlayer, handleSitOld, addOrReplaceItem, toggleBlockState, sitOn, spawnEntityRotatedByBlock, getBlockTypeVariant, decrementItemInInventory, addItemInInventory, extractColor } from './util/utils';
import { airBlocks, colors, rightBlockLocation, leftBlockLocation, backBlockLocation, frontBlockLocation, solidBlock, oppositeDirection } from './util/globalVariables';
import { PianoCalm, PianoHaggstrom, PianoMiceOnVenus, PianoSubwooferLullaby, PianoSweden, PianoWetHands } from './DecoDrop/PianoSounds';
import { sitPet } from './util/decoScripts';
import { DropUpdateSideBlock } from './DecoDrop/GlobalClass';
import { DropPaintBlock } from './DecoDrop/Paint';
import { DropCuttingBoard } from './DecoDrop/CuttingBoard';
import { DropSaloonDoor, SaloonDoor } from './DecoDrop/SaloonDoor';
import { DropKitchenTable } from './DecoDrop/KitchenTable';
import { DropCurtain } from './DecoDrop/Curtain';
import { DropBreakBlockPost, DropLampPostPlace, DropLampPostUpdate } from './DecoDrop/LampPost';
import { COOKING_POT_RECIPE_LIST, RUSTIC_OVEN_RECIPE_LIST } from './DecoDrop/Recipes';

SaloonDoor();

const DecoDropConfig = {
	accelerateTime: false,
	onlySleepAtNight: true
};

world.beforeEvents.worldInitialize.subscribe((e) => {
	e.blockComponentRegistry.registerCustomComponent("drop:candle_table", new DropCandleTable());
	e.blockComponentRegistry.registerCustomComponent("drop:sit_on_chair", new DropSitOnChair());
	e.blockComponentRegistry.registerCustomComponent("drop:sit_chair", new DropSitOnChair2());
	e.blockComponentRegistry.registerCustomComponent("drop:fence_block", new DropUpdateBlockConnections("mc:fence"));
	e.blockComponentRegistry.registerCustomComponent("drop:clay_beaker", new DropClayBeaker());
	e.blockComponentRegistry.registerCustomComponent("drop:dry_clay", new DropDryClay());
	e.blockComponentRegistry.registerCustomComponent("drop:lamp", new DropLamp());
	e.blockComponentRegistry.registerCustomComponent("drop:sit_wooden_stool", new DropSitWoodenStool());
	e.blockComponentRegistry.registerCustomComponent("drop:vase", new DropVase());
	e.blockComponentRegistry.registerCustomComponent("drop:pet_bed", new DropPetBed());
	e.blockComponentRegistry.registerCustomComponent("drop:hanging_pot", new DropHangingPot());
	e.blockComponentRegistry.registerCustomComponent("drop:bird_box", new DropBirdBox());
	e.blockComponentRegistry.registerCustomComponent("drop:add_sheet_music", new DropAddSheetMusic());
	e.blockComponentRegistry.registerCustomComponent("drop:piano_place", new DropPianoPlace());
	e.blockComponentRegistry.registerCustomComponent("drop:flatten_dough", new DropFlattenDough());
	e.blockComponentRegistry.registerCustomComponent("drop:rustic_oven", new DropRusticOven());
	e.blockComponentRegistry.registerCustomComponent("drop:add_ingredients", new DropAddIngredients());
	e.blockComponentRegistry.registerCustomComponent("drop:pizza_add_ingredients", new DropPizzaAddIngredients());
	e.blockComponentRegistry.registerCustomComponent("drop:sofa", new DropSofa());
	e.blockComponentRegistry.registerCustomComponent("drop:eat_pie", new DropEatPie());
	e.blockComponentRegistry.registerCustomComponent("drop:cut_block", new DropCutBlock());
	e.blockComponentRegistry.registerCustomComponent("drop:dyeable_cake", new DropDyeableCake());
	e.blockComponentRegistry.registerCustomComponent("drop:bowl", new DropBowl());
	e.blockComponentRegistry.registerCustomComponent("drop:frying_pan", new DropFryingPan());
	e.blockComponentRegistry.registerCustomComponent("drop:saucepan", new DropSaucepan());
	e.blockComponentRegistry.registerCustomComponent("drop:radio", new DropRadio());
	e.blockComponentRegistry.registerCustomComponent("drop:place_bed", new DropPlaceBed());
	e.blockComponentRegistry.registerCustomComponent("drop:bed_silk_touch", new DropBreakBed());
	e.blockComponentRegistry.registerCustomComponent("drop:sleep_bed", new DropBedSleep());
	e.blockComponentRegistry.registerCustomComponent("drop:sign_icon", new DropSignIcon());
	e.blockComponentRegistry.registerCustomComponent("drop:place_table_2x2", new DropPlaceTable2x2());
	e.blockComponentRegistry.registerCustomComponent("drop:table_silk_touch", new DropBreakTable2x2());
	e.blockComponentRegistry.registerCustomComponent("drop:ceramic_station", new DropCeramicStation());
	e.blockComponentRegistry.registerCustomComponent("drop:rotate_by_45", new DropRotateBy45());
	e.blockComponentRegistry.registerCustomComponent("drop:large_pot_add_dirt", new DropLargePotAddDirt());
	e.blockComponentRegistry.registerCustomComponent("drop:large_pot_plant", new DropLargePotPlant());
	e.blockComponentRegistry.registerCustomComponent("drop:break_flower", new DropBreakFlower());
	e.blockComponentRegistry.registerCustomComponent("drop:break_left_st", new DropBreakLeftBlock());
	e.blockComponentRegistry.registerCustomComponent("drop:copper_teapot", new DropCopperTeapot());
	e.blockComponentRegistry.registerCustomComponent("drop:copper_teapot_consume", new DropCopperTeapotConsume());
	e.blockComponentRegistry.registerCustomComponent("drop:toilet", new DropToilet());
	e.blockComponentRegistry.registerCustomComponent("drop:update_sided_block", new DropUpdateSideBlock("drop:floating_shelf"));
	e.blockComponentRegistry.registerCustomComponent("drop:update_sided_sofa_1", new DropSofa1("drop:sofa_1"));
	e.blockComponentRegistry.registerCustomComponent("drop:place_cabinet", new DropPlaceContainerOld("drop:cabinet_entity", "drop.ui.title.cabinet"));
	e.blockComponentRegistry.registerCustomComponent("drop:place_kitchen_cabinet", new DropPlaceContainerOld("drop:kitchen_cabinet", "drop.ui.title.kitchen_cabinet"));
	e.blockComponentRegistry.registerCustomComponent("drop:place_1x1x2_bed", new DropLargeBlockPlacer("1x1x2", { id: "drop:bed_0_entity", offset: { y: -0.2, z: 0.5 } }));
	e.blockComponentRegistry.registerCustomComponent("drop:place_2x1x2_bed", new DropLargeBlockPlacer("2x1x2", { id: "drop:bed_1", offset: { x: -0.5, z: 0.5 } }));
	e.blockComponentRegistry.registerCustomComponent("drop:place_2x1x1_block", new DropLargeBlockPlacer("2x1x1"));
	e.blockComponentRegistry.registerCustomComponent("drop:place_2x3x1_block", new DropLargeBlockPlacer("2x3x1"));
	e.blockComponentRegistry.registerCustomComponent("drop:sleep_in_bed", new DropSleepInBed());
	e.blockComponentRegistry.registerCustomComponent("drop:break_large_block", new DropBreakLargeBlock());
	e.blockComponentRegistry.registerCustomComponent("drop:place_mirror_top", new DropPlaceInvBlock("mirror"));
	e.blockComponentRegistry.registerCustomComponent("drop:place_grandfather_clock_top", new DropPlaceInvBlock("grandfather_clock"));
	e.blockComponentRegistry.registerCustomComponent("drop:ceiling_fan", new DropCeilingFan());
	e.blockComponentRegistry.registerCustomComponent("drop:water_pot", new DropWaterPot());
	e.blockComponentRegistry.registerCustomComponent("drop:random_amount_1", new DropRandomAmount(1));
	e.blockComponentRegistry.registerCustomComponent("drop:random_amount_2", new DropRandomAmount(2));
	e.blockComponentRegistry.registerCustomComponent("drop:random_amount_3", new DropRandomAmount(3));
	e.blockComponentRegistry.registerCustomComponent("drop:tomatoes_growth", new DropCrops(4, "drop:tomato", 3, "drop:tomato_seeds", 2));
	e.blockComponentRegistry.registerCustomComponent("drop:add_or_remove_pie", new DropAddPies());
	e.blockComponentRegistry.registerCustomComponent("drop:lamp_post", new DropLampPostUpdate());
	e.blockComponentRegistry.registerCustomComponent("drop:break_block_from_post", new DropBreakBlockPost());
	e.blockComponentRegistry.registerCustomComponent("drop:place_tall_window", new DropTallWindowPlace());
	e.blockComponentRegistry.registerCustomComponent("drop:tall_window_interact", new DropTallWindowInteract());
	e.blockComponentRegistry.registerCustomComponent("drop:kitchen_table", new DropKitchenTable());
	e.blockComponentRegistry.registerCustomComponent("drop:saloon_door", new DropSaloonDoor());
	e.blockComponentRegistry.registerCustomComponent("drop:fence_gate_tall", new DropFenceGateTall());
	e.blockComponentRegistry.registerCustomComponent("drop:update_curtain", new DropCurtain("drop:curtain"));
	e.blockComponentRegistry.registerCustomComponent("drop:cutting_board", new DropCuttingBoard());
	e.blockComponentRegistry.registerCustomComponent("drop:sink", new DropSink());

	e.itemComponentRegistry.registerCustomComponent("drop:paint_block", new DropPaintBlock());
	e.itemComponentRegistry.registerCustomComponent("drop:lamp_post_place", new DropLampPostPlace());
	e.itemComponentRegistry.registerCustomComponent("drop:item_break_block", new DropItemBreakBlock());
	e.itemComponentRegistry.registerCustomComponent("drop:chisel_block", new DropChiselBlock())
	e.itemComponentRegistry.registerCustomComponent("drop:consume_golden_cake", new DropConsumeGoldenCake());
	e.itemComponentRegistry.registerCustomComponent("drop:get_cooking_recipe", new DropGetCookingRecipe());
	e.itemComponentRegistry.registerCustomComponent("drop:drink_mate", new DropDrinkMate());
	e.itemComponentRegistry.registerCustomComponent("drop:drink_milk", new DropDrinkMilk());
	e.itemComponentRegistry.registerCustomComponent("drop:start_config", new DropGlobalConfig());
	e.itemComponentRegistry.registerCustomComponent("drop:place_water_pot", new DropPlaceWaterPot());
});
const directionState = "minecraft:cardinal_direction";
class DropSink {
	onPlayerInteract(e) {
		const { player, block, dimension } = e;
		const direction = block.permutation.getState(directionState);
		const center = block.center();
		dimension.spawnParticle("drop:water_from_sink", {
			x: center.x + (direction === "west" ? 0.12 : direction === "east" ? -0.12 : 0),
			y: center.y + 0.6,
			z: center.z + (direction === "north" ? 0.12 : direction === "south" ? -0.12 : 0)
		});
		const item = player.getComponent("equippable")?.getEquipment(EquipmentSlot.Mainhand);
		const hasBucket = item?.matches("minecraft:bucket");
		const waterLevel = block.permutation.getState("drop:water_level");
		if (waterLevel < 4 && !hasBucket) {
			block.setPermutation(block.permutation.withState("drop:water_level", waterLevel + 1));
			dimension.playSound("liquid.water", center);
			dimension.playSound("bucket.empty_water", center);
		} else if (waterLevel > 0 && hasBucket) {
			block.setPermutation(block.permutation.withState("drop:water_level", waterLevel - 1));
			dimension.spawnItem(new ItemStack("minecraft:water_bucket", 1), center);
			decrementItemInHand(player);
			dimension.playSound("liquid.water", center);
			dimension.playSound("bucket.fill_water", center);
		}
	}
}
class DropFenceGateTall {
	constructor() {
		this.WOOD_SOUNDS = {
			"bamboo": { close: "close.bamboo_wood_fence_gate", open: "open.bamboo_wood_fence_gate" },
			"cherry": { close : "close.cherry_wood_fence_gate", open: "open.cherry_wood_fence_gate" },
			"nether": { close: "close.nether_wood_fence_gate", open: "open.nether_wood_fence_gate" },
			"default": { close: "close.fence_gate", open: "open.fence_gate" }
		}
		this.onPlayerInteract = this.onPlayerInteract.bind(this);
		this.beforeOnPlayerPlace = this.beforeOnPlayerPlace.bind(this);
	}
	updateTopBlock(block, open) {
		const top = block.above();
		if (open && top.matches("drop:fence_top")) top.setType("minecraft:air");
		else if (!open && top.isAir) top.setType("drop:fence_top");
	}
	onPlayerInteract(e) {
		const { block, dimension } = e;
		const direction = block.permutation.getState(directionState);
		const open = block.permutation.getState("drop:open");
		const newOpenState = !open;
		block.setPermutation(block.permutation.withState("drop:open", newOpenState));
		const sound = this.WOOD_SOUNDS[getBlockTypeVariant(block.typeId)];
		dimension.playSound(open ? sound.close : sound.open, block.center());
		const isSideBlock = block.permutation.getState("drop:side");
		const sideBlock = isSideBlock ? block.offset(leftBlockLocation[direction]) : block.offset(rightBlockLocation[direction]);
		this.updateTopBlock(block, newOpenState);
		if (sideBlock.hasTag("drop:is_fence_gate")) {
			sideBlock.setPermutation(sideBlock.permutation.withState("drop:open", newOpenState));
			this.updateTopBlock(sideBlock, newOpenState);
		}
	}
	onPlace(e) {
		const blockAbove = e.block.above();
		if (!blockAbove.isAir) return;
		blockAbove.setType("drop:fence_top");
	}
	beforeOnPlayerPlace(e) {
		const { block, permutationToPlace } = e;
		const direction = permutationToPlace.getState(directionState);
		const isRightState = block.offset(leftBlockLocation[direction]).hasTag("drop:side_fence_gate");
		e.permutationToPlace = permutationToPlace.withState("drop:side", isRightState)
	}
}
class DropChiselBlock {
	onUseOn(e) {
		const { block, itemStack, source } = e;
		if (!block.hasTag("drop:chiselable")) return;
		const id = block.typeId;
		const lastUnderscore = id.lastIndexOf('_');
		if (lastUnderscore === -1) return;
		const baseId = id.substring(0, lastUnderscore);
		const currentState = parseInt(id.substring(lastUnderscore + 1));
		block.setType(`${baseId}_${(currentState + 1) % (5 + 1)}`);
		source.dimension.playSound("effect.chisel_block", block.center());
		itemApplyDamage(source, itemStack);
	}
}
class DropAddPies {
	constructor() {
		this.onPlayerInteract = this.onPlayerInteract.bind(this);
		this.onPlayerDestroy = this.onPlayerDestroy.bind(this)
	}
	getItemName(id) {
		return id.replace("stack", "block") + "_item";
	}
	updateAmount(block, amount, sound) {
		block.setPermutation(block.permutation.withState("drop:amount", amount));
		block.dimension.playSound(sound, block.center());
	}
	onPlayerInteract(e){
		const { player, block, dimension } = e;
		const item = player.getComponent("equippable")?.getEquipment(EquipmentSlot.Mainhand);
		const amount = block.permutation.getState("drop:amount");
		if (this.getItemName(block.typeId) === item?.typeId && amount < 4) {
			decrementItemInHand(player);
			this.updateAmount(block, amount + 1, "block.itemframe.add_item");
		} else if (this.getItemName(block.typeId) !== item?.typeId && amount >= 2) {
			dimension.spawnItem(new ItemStack(this.getItemName(block.typeId), 1), block.bottomCenter());
			if (amount > 2) {
				this.updateAmount(block, amount - 1, "block.itemframe.remove_item");
			} else {
				const direction = block.permutation.getState(directionState);
				block.setPermutation(BlockPermutation.resolve(block.typeId.replace("stack", "block"), { [directionState]: direction }));
				dimension.playSound("block.itemframe.remove_item", block.center());
			}
		}
	}
	onPlayerDestroy(e) {
		const { destroyedBlockPermutation, block, dimension } = e;
		dimension.spawnItem(new ItemStack(this.getItemName(destroyedBlockPermutation.type.id), destroyedBlockPermutation.getState("drop:amount")), block.bottomCenter());
	}
}
const COOKING_TIME = 15;
const SERVING_TIME = 1.5;
const UPDATE_INTERVAL = 0.1;
const MAX_TIMER_STACK = 64;
const LIQUID_INPUTS = {
	"minecraft:water_bucket": { convertTo: "minecraft:bucket", resultItem: "drop:liquid_teapot_water", blockState: "water" },
	"minecraft:milk_bucket": { convertTo: "minecraft:bucket", resultItem: "drop:liquid_teapot_milk", blockState: "milk" }
};
const SERVABLE_LIQUIDS = {
	"drop:liquid_teapot_water": "drop:cup_water",
	"drop:liquid_teapot_milk": "drop:cup_milk",
	"drop:liquid_teapot_mate": "drop:cup_mate"
};
const INFUSION_LEAVES = {
	"drop:mate_leaf": { resultItem: "drop:liquid_teapot_mate", blockState: "mate" }
}
const isValidLiquidInput = (item) => item && LIQUID_INPUTS[item.typeId];
const timerItem = (amount) => new ItemStack("drop:timer", Math.max(1, Math.min(64, amount)));
const setTimer = (inventory, slot, progress, duration) => {
	const ratio = progress / duration;
	const timerAmount = Math.floor(ratio * MAX_TIMER_STACK);
	inventory.setItem(slot, timerItem(timerAmount));
};
const COOKING_POT_RECIPES = new Map();
for (const recipe of COOKING_POT_RECIPE_LIST) {
	const key = JSON.stringify(trimGrid(recipe.grid));
	COOKING_POT_RECIPES.set(key, {
		result: recipe.result,
		amount: recipe.amount,
		serving: recipe.serving ?? true
	});
}
const RUSTIC_OVEN_RECIPES = new Map();
for (const recipe of RUSTIC_OVEN_RECIPE_LIST) {
	const key = JSON.stringify(trimGrid(recipe.grid));
	RUSTIC_OVEN_RECIPES.set(key, {
		result: recipe.result,
		amount: recipe.amount
	});
}
class DropRusticOven{
	onPlace(e) {
		const { block, dimension } = e;
		const cookingPot = dimension.spawnEntity("drop:rustic_oven", block.bottomCenter());
		cookingPot.nameTag = "drop.ui.title.rustic_oven";
		cookingPot.addEffect("invisibility", 10, { showParticles: false });
		const inventory = cookingPot.getComponent("inventory").container;
		inventory.setItem(10, timerItem(64));
		inventory.setItem(11, timerItem(1));
	}
}
function canCookOnOven(inventory) {
	const grid = [
		[inventory.getItem(0)?.typeId ?? "", inventory.getItem(1)?.typeId ?? "", inventory.getItem(2)?.typeId ?? ""],
		[inventory.getItem(3)?.typeId ?? "", inventory.getItem(4)?.typeId ?? "", inventory.getItem(5)?.typeId ?? ""],
		[inventory.getItem(6)?.typeId ?? "", inventory.getItem(7)?.typeId ?? "", inventory.getItem(8)?.typeId ?? ""]
	];
	return isValidRecipe(grid, RUSTIC_OVEN_RECIPES);
}
const RUSTIC_OVEN_COOKING_TIME = 15;
world.afterEvents.dataDrivenEntityTrigger.subscribe(({ entity, eventId }) => {
	if (!entity) return;
	const inventory = entity.getComponent("inventory").container;
	const cookingPotData = JSON.parse(entity.getDynamicProperty("drop:rustic_oven_data") || JSON.stringify({
		cookProgress: 0,
		canCook: false,
		recipeKey: undefined,
		recipeResult: undefined,
		recipeAmount: undefined
	}));
	const isHeatSource = entity.getProperty("drop:is_on_oven");
	if (eventId === "drop:check") {
		const validRecipe = canCookOnOven(inventory);
		if (isHeatSource && validRecipe) {
			entity.triggerEvent("drop:start");
			cookingPotData.canCook = true;
			cookingPotData.recipeKey = getInvKey(inventory, 8);
			cookingPotData.recipeResult = validRecipe.result;
			cookingPotData.recipeAmount = validRecipe.amount;
			entity.setDynamicProperty("drop:rustic_oven_data", JSON.stringify(cookingPotData));
		}
		return;
	}
	if (eventId === "drop:update_timer") {
		if (!isHeatSource) {
			cookingPotData.cookProgress = 0;
			inventory.setItem(11, timerItem(1));
			entity.setDynamicProperty("drop:rustic_oven_data", JSON.stringify(cookingPotData));
			return;
		}
		const newRecipeKey = getInvKey(inventory, 8);
		if (newRecipeKey !== cookingPotData.recipeKey) {
			const validRecipe = canCookOnOven(inventory);
			cookingPotData.recipeKey = newRecipeKey;
			cookingPotData.cookProgress = 0;
			inventory.setItem(11, timerItem(1));
			if (validRecipe) {
				cookingPotData.canCook = true;
				cookingPotData.recipeResult = validRecipe.result;
				cookingPotData.recipeAmount = validRecipe.amount;
			} else {
				cookingPotData.canCook = false;
				cookingPotData.recipeResult = undefined;
				cookingPotData.recipeAmount = undefined;
			}
		}
		if (!cookingPotData.canCook) {
			cookingPotData.cookProgress = 0;
			inventory.setItem(11, timerItem(1));
			entity.setDynamicProperty("drop:rustic_oven_data", JSON.stringify(cookingPotData));
			return;
		}
		if (cookingPotData.cookProgress < RUSTIC_OVEN_COOKING_TIME) {
			const resultSlot = inventory.getItem(9);
			if (!resultSlot || resultSlot.typeId === cookingPotData.recipeResult && (resultSlot.amount + cookingPotData.recipeAmount <= resultSlot.maxAmount)) {
				cookingPotData.cookProgress += UPDATE_INTERVAL;
				setTimer(inventory, 11, cookingPotData.cookProgress, RUSTIC_OVEN_COOKING_TIME);
			} else {
				cookingPotData.cookProgress = 0;
				inventory.setItem(11, timerItem(1));
			}
		} else {
			const resultSlot = inventory.getItem(9);
			if (!resultSlot || resultSlot.matches("drop:timer")) {
				inventory.setItem(9, new ItemStack(cookingPotData.recipeResult, cookingPotData.recipeAmount));
			} else if (resultSlot.typeId === cookingPotData.recipeResult) {
				resultSlot.amount += cookingPotData.recipeAmount;
				inventory.setItem(9, resultSlot);
			}
			for (let i = 0; i <= 8; i++) {
				decrementItemInInventory(entity, i);
			}
			cookingPotData.cookProgress = 0;
			inventory.setItem(11, timerItem(1));
			const validRecipe = canCookOnOven(inventory);
			if (validRecipe) {
				cookingPotData.canCook = true;
				cookingPotData.recipeResult = validRecipe.result;
				cookingPotData.recipeAmount = validRecipe.amount;
				cookingPotData.recipeKey = getInvKey(inventory, 8);
			} else {
				cookingPotData.canCook = false;
				cookingPotData.recipeResult = undefined;
				cookingPotData.recipeAmount = undefined;
				cookingPotData.recipeKey = undefined;
			}
		}
		entity.setDynamicProperty("drop:rustic_oven_data", JSON.stringify(cookingPotData));
	}
}, {
	entityTypes: [ "drop:rustic_oven" ],
	eventTypes: [ "drop:check", "drop:update_timer" ]
})
class DropSaucepan {
	onPlayerInteract(e) { return; }
	onPlace(e) {
		const { block, dimension } = e;
		const cookingPot = dimension.spawnEntity("drop:cooking_pot", block.bottomCenter());
		cookingPot.nameTag = "drop.ui.title.cooking_pot";
		cookingPot.addEffect("invisibility", 10, { showParticles: false });
		const inventory = cookingPot.getComponent("inventory").container;
		inventory.setItem(9, timerItem(1));
		inventory.setItem(11, timerItem(1));
		inventory.setItem(12, timerItem(1));
	}
}
function trimGrid(grid) {
	grid = grid.filter(row => row.some(cell => cell !== ""));
	if (grid.length === 0) return [[]];
	const numCols = grid[0].length;
	const colsToKeep = [];
	for (let col = 0; col < numCols; col++) {
		if (grid.some(row => row[col] !== "")) colsToKeep.push(col);
	}
	const trimmed = grid.map(row => colsToKeep.map(col => row[col]));
	return trimmed;
}
function isValidRecipe(grid, RECIPES_MAP) {
	const trimmed = trimGrid(grid);
	const key = JSON.stringify(trimmed);
	return RECIPES_MAP.get(key);
}
function getInvKey(inventory, maxSlot) {
	const items = [];
	for (let i = 0; i <= maxSlot; i++) {
		items.push(inventory.getItem(i)?.typeId);
	}
	return items.join("|");
}
function canServeFood(inventory) {
	const resultSlot = inventory.getItem(9);
	const resultTypeId = resultSlot?.typeId;
	if (!resultTypeId) return false;
	const lore = resultSlot?.getLore?.();
	if (lore?.length) {
		const value = lore[0];
		const hasServeIngredient = value === "true" || inventory.getItem(7)?.matches(value);
		if (!hasServeIngredient) return false;
		const outputSlot = inventory.getItem(8);
		const canAddOutput = !outputSlot || (outputSlot.matches(resultTypeId) && outputSlot.amount < outputSlot.maxAmount);
		return canAddOutput;
	}
	return false;
}
function getServeKey(inventory) {
	const s6 = inventory.getItem(7);
	const s7 = inventory.getItem(8);
	const s8 = inventory.getItem(9);

	return `${s6?.typeId}_${s7?.typeId}_${s8?.typeId}`
}
function canCook(inventory) {
	const grid = [
		[inventory.getItem(0)?.typeId ?? "", inventory.getItem(1)?.typeId ?? "", inventory.getItem(2)?.typeId ?? ""],
		[inventory.getItem(3)?.typeId ?? "", inventory.getItem(4)?.typeId ?? "", inventory.getItem(5)?.typeId ?? ""]
	];
	return isValidRecipe(grid, COOKING_POT_RECIPES);
}
function resetCookingTimers(entity, cookingPotData, inventory) {
	cookingPotData.cookProgress = 0;
	cookingPotData.serveProgress = 0;
	inventory.setItem(11, timerItem(1));
	inventory.setItem(12, timerItem(1));
	entity.setDynamicProperty("drop:cooking_pot_data", JSON.stringify(cookingPotData));
}
world.afterEvents.dataDrivenEntityTrigger.subscribe(({ entity, eventId }) => {
	if (!entity) return;
	const inventory = entity.getComponent("inventory").container;
	const cookingPotData = JSON.parse(entity.getDynamicProperty("drop:cooking_pot_data") || JSON.stringify({
		cookProgress: 0,
		serveProgress: 0,
		canCook: false,
		recipeKey: undefined,
		recipeResult: undefined,
		recipeAmount: undefined,
		canServe: false,
		serveKey: undefined,
	}));
	const isHeatSource = entity.getProperty("drop:is_on_oven");
	if (eventId === "drop:check") {
		const validRecipe = canCook(inventory);
		const canServe = canServeFood(inventory);
		const shouldBeActive = isHeatSource && (validRecipe || canServe);
		if (shouldBeActive) {
			entity.triggerEvent("drop:start");
			cookingPotData.serveKey = getServeKey(inventory);
			if (canServe) cookingPotData.canServe = true;
			if (validRecipe) {
				cookingPotData.canCook = true;
				cookingPotData.recipeResult = validRecipe.result;
				cookingPotData.recipeAmount = validRecipe.amount;
			}
			entity.setDynamicProperty("drop:cooking_pot_data", JSON.stringify(cookingPotData));
		}
		return;
	}
	if (eventId === "drop:update_timer") {
		if (!isHeatSource) {
			resetCookingTimers(entity, cookingPotData, inventory);
			return;
		}
		const newServeKey = getServeKey(inventory);
		if (newServeKey !== cookingPotData.serveKey) {
			cookingPotData.canServe = canServeFood(inventory);
			cookingPotData.serveKey = newServeKey;
			cookingPotData.serveProgress = 0;
			inventory.setItem(12, timerItem(1));
		}
		const newRecipeKey = getInvKey(inventory, 5);
		if (newRecipeKey !== cookingPotData.recipeKey) {
			const validRecipe = canCook(inventory);
			cookingPotData.recipeKey = newRecipeKey;
			cookingPotData.cookProgress = 0;
			inventory.setItem(11, timerItem(1));
			if (validRecipe) {
				cookingPotData.canCook = true;
				cookingPotData.recipeResult = validRecipe.result;
				cookingPotData.recipeAmount = validRecipe.amount;
				cookingPotData.serving = validRecipe.serving;
			} else {
				cookingPotData.canCook = false;
				cookingPotData.recipeResult = undefined;
				cookingPotData.recipeAmount = undefined;
			}
		}
		if (!cookingPotData.canCook && !cookingPotData.canServe) {
			entity.triggerEvent("drop:stop");
			resetCookingTimers(entity, cookingPotData, inventory);
			return;
		}
		if (cookingPotData.canCook) {
			const resultSlot = inventory.getItem(9);
			if (cookingPotData.cookProgress < COOKING_TIME) {
				if (!resultSlot) return inventory.setItem(9, timerItem(1));
				const newResultAmount = resultSlot?.amount + cookingPotData.recipeAmount;
				if (resultSlot?.matches("drop:timer") ||
					(resultSlot?.typeId === cookingPotData.recipeResult && newResultAmount <= resultSlot.maxAmount)) {

					cookingPotData.cookProgress += UPDATE_INTERVAL;
					setTimer(inventory, 11, cookingPotData.cookProgress, COOKING_TIME);
				} else {
					cookingPotData.cookProgress = 0;
					inventory.setItem(11, timerItem(1));
				}
			} else {
				const validRecipe = canCook(inventory);
				if (!resultSlot || resultSlot.matches("drop:timer")) {
					const res = new ItemStack(cookingPotData.recipeResult, cookingPotData.recipeAmount);
					res.setLore([`${validRecipe.serving}`]);
					inventory.setItem(9, res);
				} else if (resultSlot?.typeId === cookingPotData.recipeResult) {
					resultSlot.amount += cookingPotData.recipeAmount;
					inventory.setItem(9, resultSlot);
				}
				for (let i = 0; i <= 5; i++) {
					decrementItemInInventory(entity, i);
				}
				cookingPotData.cookProgress = 0;
				inventory.setItem(11, timerItem(1));
				cookingPotData.recipeKey = newRecipeKey;
				if (validRecipe) {
					cookingPotData.canCook = true;
					cookingPotData.recipeResult = validRecipe.result;
					cookingPotData.recipeAmount = validRecipe.amount;
					cookingPotData.serving = validRecipe.serving;
				} else {
					cookingPotData.canCook = false;
					cookingPotData.recipeResult = undefined;
					cookingPotData.recipeAmount = undefined;
					cookingPotData.serving = undefined;
				}
			}
		}
		if (cookingPotData.canServe) {
			const resultSlot = inventory.getItem(9);
			const output = inventory.getItem(8);
			if (output && (output.amount >= output.maxAmount)) {
				cookingPotData.serveProgress = 0;
				inventory.setItem(12, timerItem(1));
			} else {
				cookingPotData.serveProgress += UPDATE_INTERVAL;
				setTimer(inventory, 12, cookingPotData.serveProgress, SERVING_TIME);
				if (cookingPotData.serveProgress >= SERVING_TIME) {
					decrementItemInInventory(entity, 9, { convertTo: new ItemStack("drop:timer", 1) });
					const lore = resultSlot?.getLore?.();
					if (lore?.[0] && lore[0] !== "true") decrementItemInInventory(entity, 7);
					addItemInInventory(inventory, 8, new ItemStack(resultSlot.typeId, 1));
					entity.dimension.playSound("random.potion.brewed", entity.location);
					cookingPotData.serveProgress = 0;
					inventory.setItem(12, timerItem(1));
					cookingPotData.canServe = canServeFood(inventory);
					cookingPotData.serveKey = getServeKey(inventory);
				}
			}
		}
		entity.setDynamicProperty("drop:cooking_pot_data", JSON.stringify(cookingPotData));
	}
}, {
	entityTypes: [ "drop:cooking_pot" ],
	eventTypes: [ "drop:check", "drop:update_timer" ]
});
class DropCopperTeapot {
	onPlace(e) {
		const { block, dimension } = e;
		const teapotEntity = dimension.spawnEntity("drop:teapot", block.bottomCenter());
		teapotEntity.nameTag = "drop.ui.title.teapot_copper";
		teapotEntity.addEffect("invisibility", 10, { showParticles: false });
		const inventory = teapotEntity.getComponent("inventory").container;
		inventory.setItem(4, timerItem(1));
		inventory.setItem(6, timerItem(1));
		inventory.setItem(7, timerItem(1));
	}
	onPlayerInteract(e) { return; }
}
world.afterEvents.dataDrivenEntityTrigger.subscribe(({ entity, eventId }) => {
	if (!entity) return;
	try {
		const inventory = entity.getComponent("inventory").container;
		const teapotData = JSON.parse(entity.getDynamicProperty("drop:teapot_data") || JSON.stringify({
			cookProgress: 0,
			serveProgress: 0,
			isActive: false,
			usedLiquid: false,
			filledResult: false
		}));
		const isHeatSource = entity.getProperty("drop:is_on_oven");
		if (eventId === "drop:check") {
			const shouldBeActive = isHeatSource && isValidLiquidInput(inventory.getItem(1)) && !teapotData.usedLiquid;
			if (shouldBeActive) {
				entity.triggerEvent("drop:start");
				teapotData.isActive = true;
			}
			entity.setDynamicProperty("drop:teapot_data", JSON.stringify(teapotData));
			return;
		}
		if (eventId === "drop:update_timer") {
			if (!isHeatSource) {
				teapotData.cookProgress = 0;
				teapotData.serveProgress = 0;
				inventory.setItem(6, timerItem(1));
				inventory.setItem(7, timerItem(1));
				entity.setDynamicProperty("drop:teapot_data", JSON.stringify(teapotData));
				return;
			}
			const input = inventory.getItem(1);
			const cup = inventory.getItem(2);
			const output = inventory.getItem(3);
			const resultSlot = inventory.getItem(4);

			if (!teapotData.usedLiquid && isValidLiquidInput(input) && resultSlot?.matches("drop:timer")) {
				decrementItemInInventory(entity, 1, {
					amount: 1,
					convertTo: new ItemStack(LIQUID_INPUTS[input.typeId].convertTo, 1)
				});
				inventory.setItem(4, new ItemStack(LIQUID_INPUTS[input.typeId].resultItem, 1));
				teapotData.cookProgress = 0;
				teapotData.serveProgress = 0;
				teapotData.usedLiquid = true;
				const teapotBlock = entity.dimension.getBlock(entity.location)
				teapotBlock.setPermutation(teapotBlock.permutation.withState("drop:food", LIQUID_INPUTS[input.typeId].blockState));
			}
			if (resultSlot) {
				if (teapotData.cookProgress < COOKING_TIME && SERVABLE_LIQUIDS[resultSlot?.typeId]) {
					teapotData.cookProgress += UPDATE_INTERVAL;
					setTimer(inventory, 6, teapotData.cookProgress, COOKING_TIME);
					const leaf = inventory.getItem(0);
					if (resultSlot.matches("drop:liquid_teapot_water") && leaf && INFUSION_LEAVES[leaf.typeId]) {
						inventory.setItem(4, new ItemStack(INFUSION_LEAVES[leaf.typeId].resultItem, 1));
						decrementItemInInventory(entity, 0);
						const teapotBlock = entity.dimension.getBlock(entity.location)
						teapotBlock.setPermutation(teapotBlock.permutation.withState("drop:food", INFUSION_LEAVES[leaf.typeId].blockState));
					}
				}
				if (teapotData.cookProgress >= COOKING_TIME) {
					if (!teapotData.filledResult) {
						inventory.setItem(4, new ItemStack(resultSlot.typeId, 3));
						teapotData.filledResult = true;
					}
					if (cup?.matches("drop:cup_empty")) {
						if (output && (output.amount >= output.maxAmount || !output.matches(SERVABLE_LIQUIDS[resultSlot.typeId]))) {
							teapotData.serveProgress = 0;
							inventory.setItem(7, timerItem(1));
						} else {
							teapotData.serveProgress += UPDATE_INTERVAL;
							setTimer(inventory, 7, teapotData.serveProgress, SERVING_TIME);

							if (teapotData.serveProgress >= SERVING_TIME) {
								decrementItemInInventory(entity, 2);
								decrementItemInInventory(entity, 4);
								entity.dimension.playSound("random.potion.brewed", entity.location);
								addItemInInventory(inventory, 3, new ItemStack(SERVABLE_LIQUIDS[resultSlot.typeId], 1));
								teapotData.serveProgress = 0;
								inventory.setItem(7, timerItem(1));
							}
						}
					} else {
						teapotData.serveProgress = 0;
						inventory.setItem(7, timerItem(1));
					}
				}
			} else {
				teapotData.usedLiquid = false;
				teapotData.filledResult = false;
				teapotData.isActive= false;
				teapotData.cookProgress = 0;
				teapotData.serveProgress = 0;
				inventory.setItem(4, timerItem(1));
				inventory.setItem(6, timerItem(1));
				inventory.setItem(7, timerItem(1));
				const teapotBlock = entity.dimension.getBlock(entity.location)
				teapotBlock.setPermutation(teapotBlock.permutation.withState("drop:food", "empty"));
				entity.triggerEvent("drop:stop");
			}
			entity.setDynamicProperty("drop:teapot_data", JSON.stringify(teapotData));
		}
	} catch (error) {}
}, {
	entityTypes: [ "drop:teapot" ],
	eventTypes: [ "drop:check", "drop:update_timer" ]
});
class DropCrops {
	constructor(maxStage, cropId, cropMaxAmount, seedId, seedMaxAmount) {
		this.maxStage = maxStage;
		this.cropId = cropId;
		this.cropMaxAmount = cropMaxAmount;
		this.seedId = seedId;
		this.seedMaxAmount = seedMaxAmount;
		this.growthStage = "mc:growth_stage";
		this.onRandomTick = this.onRandomTick.bind(this);
		this.onPlayerInteract = this.onPlayerInteract.bind(this);
	}
	onRandomTick(e) {
		const { block } = e;
		const growthStage = block.permutation.getState(this.growthStage);
		if (Math.random() > 0.3 || growthStage === this.maxStage) return;
		this.setGrowthStage(block, growthStage + 1, "place.moss");
	}
	onPlayerInteract(e) {
		const { player, block, dimension } = e;
		const growthStage = block.permutation.getState(this.growthStage);
		const item = player.getComponent("equippable").getEquipment(EquipmentSlot.Mainhand);
		if (growthStage < this.maxStage && item?.typeId === "minecraft:bone_meal") {
			this.setGrowthStage(block, growthStage + 1, "item.bone_meal.use");
			decrementItemInHand(player);
			dimension.spawnParticle("minecraft:crop_growth_emitter", block.center());
		} else if (growthStage === this.maxStage) {
			const getRandomAmount = (max) => Math.floor(Math.random() * max) + 1;
			dimension.spawnItem(new ItemStack(this.cropId, getRandomAmount(this.cropMaxAmount)), block.center());
			dimension.spawnItem(new ItemStack(this.seedId, getRandomAmount(this.seedMaxAmount)), block.center());
			this.setGrowthStage(block, 0, "place.moss");
		}
	}
	setGrowthStage(block, growthStage, sound) {
		block.setPermutation(block.permutation.withState(this.growthStage, growthStage));
		block.dimension.playSound(sound, block.center());
	}
}
class DropRandomAmount {
	constructor(maxAmount) {
		this.maxAmount = maxAmount + 1;
		this.beforeOnPlayerPlace = this.beforeOnPlayerPlace.bind(this);
	}
	beforeOnPlayerPlace(e) {
		e.permutationToPlace = e.permutationToPlace.withState("drop:amount", Math.floor(Math.random() * this.maxAmount));
	}
}
const liquids = [ "minecraft:water_bucket", "minecraft:milk_bucket" ];
class DropPlaceWaterPot {
	onUseOn(e) {
		const { source, block, blockFace, itemStack } = e;
		if (blockFace !== Direction.Up) return;
		const aboveBlock = block.above();
		if (!airBlocks.includes(aboveBlock.typeId)) return;
		let props = itemStack.getDynamicProperty("drop:liquid_properties");
		if (!props) return;
		props = JSON.parse(props);
		aboveBlock.setPermutation(BlockPermutation.resolve(itemStack.typeId.replace("_liquid", ""), { "drop:liquid_type": props["drop:liquid_type"] }));
		const waterPotEntity = block.dimension.spawnEntity("drop:water_pot", aboveBlock.center());
		waterPotEntity.setProperty("drop:liquid_type", props["drop:liquid_type"]);
		waterPotEntity.setProperty("drop:liquid_amount", props["drop:liquid_amount"]);
		decrementItemInHand(source, true);
		block.dimension.playSound("break.decorated_pot", aboveBlock.center());
	}
}
class DropWaterPot {
	constructor() {
		this.liquidState = "drop:liquid_type";
		this.liquidAmountState = "drop:liquid_amount";
		this.maxAmount = 32;
		this.onPlayerInteract = this.onPlayerInteract.bind(this);
		this.onPlayerDestroy = this.onPlayerDestroy.bind(this);
	}
	fillBucket(player, dimension, block, fluidEntity) {
		if (!fluidEntity) return;
		const liquidAmount = fluidEntity.getProperty(this.liquidAmountState);
		if (liquidAmount <= 0) return;
		const liquidType = fluidEntity.getProperty(this.liquidState);
		dimension.playSound("bucket.fill_water", block.center());
		addOrReplaceItem(player, liquids[liquidType - 1], 1);
		const newAmount = liquidAmount - 1;
		this.updateLiquidAmount(fluidEntity, block, newAmount, player);
	}
	addLiquid(player, dimension, block, fluidEntity, item) {
		const liquidValue = liquids.indexOf(item.typeId) + 1;
		if (fluidEntity && fluidEntity.getProperty(this.liquidState) !== liquidValue) return;
		const currentAmount = fluidEntity?.getProperty(this.liquidAmountState) || 0;
		if (currentAmount >= this.maxAmount) return player.onScreenDisplay.setActionBar("Ya no se puede agregar más");
		if (!fluidEntity) fluidEntity = this.createNewFluidEntity(dimension, block, liquidValue);
		decrementItemInInventory(player, player.selectedSlotIndex, { convertTo: new ItemStack("minecraft:bucket", 1) });
		dimension.playSound("bucket.empty_water", block.center());
		this.updateLiquidAmount(fluidEntity, block, currentAmount + 1, player);
	}
	createNewFluidEntity(dimension, block, liquidValue) {
		const entity = dimension.spawnEntity("drop:water_pot", block.center());
		entity.setProperty(this.liquidState, liquidValue);
		entity.setProperty(this.liquidAmountState, 1);
		block.setPermutation(block.permutation.withState(this.liquidState, liquidValue));
		return entity;
	}
	updateLiquidAmount(fluidEntity, block, amount, player) {
		fluidEntity.setProperty(this.liquidAmountState, amount);
		player.onScreenDisplay.setActionBar(`${amount}/${this.maxAmount}`);
		if (amount === 0) {
			fluidEntity.remove();
			block.setPermutation(block.permutation.withState(this.liquidState, 0));
		}
	}
	onPlayerInteract(e) {
		const { player, block, dimension } = e;
		const item = player.getComponent("equippable").getEquipment(EquipmentSlot.Mainhand);
		if (!(item && (liquids.includes(item.typeId) || item?.typeId === "minecraft:bucket"))) return;
		const fluidEntity = dimension.getEntitiesAtBlockLocation(block.location).find(entity => entity.typeId === "drop:water_pot");
		if (item.typeId === "minecraft:bucket") this.fillBucket(player, dimension, block, fluidEntity);
		else this.addLiquid(player, dimension, block, fluidEntity, item);
	}
	onPlayerDestroy(e) {
		const { player, block, dimension, destroyedBlockPermutation } = e;
		const item = player.getComponent("equippable").getEquipment(EquipmentSlot.Mainhand);
		const hasSilkTouch = item?.getComponent(ItemEnchantableComponent.componentId)?.getEnchantment("silk_touch");
		if (item && hasSilkTouch) return;
		const fluidEntity = dimension.getEntitiesAtBlockLocation(block.location).find(entity => entity.typeId === "drop:water_pot");
		if (!fluidEntity) {
			if (destroyedBlockPermutation.getState(this.liquidState) > 0 && !player.matches({ gameMode: GameMode.creative })) dimension.spawnItem(new ItemStack(destroyedBlockPermutation.type.id, 1), block.center());
			return;
		}
		const liquidType = fluidEntity.getProperty(this.liquidState);
		const liquidAmount = fluidEntity.getProperty(this.liquidAmountState);
		const dropFullItem = new ItemStack(`${destroyedBlockPermutation.type.id}_liquid`, 1);
		dropFullItem.setDynamicProperty("drop:liquid_properties", JSON.stringify({
			[this.liquidState]: liquidType,
			[this.liquidAmountState]: liquidAmount
		}))
		dropFullItem.setLore([
			`§fLiquid: §7${liquidType === 1 ? "Water" : "Milk"}`,
			`§fAmount: §7${liquidAmount}`
		])
		dimension.spawnItem(dropFullItem, block.center());
	}
}
class DropCeilingFan {
	constructor() {
		this.ceilingFanId = "drop:ceiling_fan";
		this.litState = "drop:lit"
		this.onPlace = this.onPlace.bind(this);
		this.onPlayerInteract = this.onPlayerInteract.bind(this);
	}
	onPlace(e) {
		e.dimension.spawnEntity(this.ceilingFanId, e.block.center()).triggerEvent(e.block.typeId);
	}
	onPlayerInteract(e) {
		const { block, dimension } = e;
		const cf = dimension.getEntitiesAtBlockLocation(block.location).find(e => e.typeId === this.ceilingFanId);
		if (!cf) return;
		const lit = block.permutation.getState(this.litState);
		block.setPermutation(block.permutation.withState(this.litState, !lit));
		cf.setProperty(this.litState, !lit);
		dimension.playSound("random.lever_click", block.center());
	}
}
class DropSofa1 extends DropUpdateSideBlock {
	constructor(blockTag) {
		super(blockTag);
		this.onPlayerInteract = this.onPlayerInteract.bind(this);
	}
	onPlayerInteract(e) {
		sitOn(e.block, e.player, "drop:chair_entity_drop", { y: -0.4 });
	}
}
class DropPlaceInvBlock {
	constructor(blockType) {
		this.blockType = blockType;
		this.onPlace = this.onPlace.bind(this);
		this.beforeOnPlayerPlace = this.beforeOnPlayerPlace.bind(this);
	}
	onPlace(e) {
		const { block } = e;
		block.above().setPermutation(BlockPermutation.resolve(`drop:${this.blockType}_${getBlockTypeVariant(block.typeId)}_top`, { [directionState]: block.permutation.getState(directionState) }));
		block.setPermutation(block.permutation.withState("drop:placed", true));
	}
	beforeOnPlayerPlace(e) {
		if (!airBlocks.includes(e.block.above().typeId)) e.cancel = true;
	}
}
const blockPartGetters = new Map([
	["2x1x1", (direction, block) => [
		{ target: block.offset(rightBlockLocation[direction]), part: 2 }
	]],
	["1x1x2", (direction, block) => [
		{ target: block.offset(backBlockLocation[direction]), part: 2 }
	]],
	["1x2x1", (direction, block) => [
		{ target: block.above(), part: 2 }
	]],
	["2x1x2", (direction, block) => {
		const right = block.offset(rightBlockLocation[direction]);
		return [
			{ target: right, part: 2 },
			{ target: right.offset(backBlockLocation[direction]), part: 3 },
			{ target: block.offset(backBlockLocation[direction]), part: 4 }
		];
	}],
	["2x3x1", (direction, block) => {
		const blockAbove = block.above();
		const right = block.offset(rightBlockLocation[direction]);
		const rightAbove = right.above();
		return [
			{ target: blockAbove, part: 2 },
			{ target: blockAbove.above(), part: 3 },
			{ target: rightAbove.above(), part: 4 },
			{ target: rightAbove, part: 5 },
			{ target: right, part: 6 }
		];
	}],
	["2x2x2", (direction, block) => {
		const right = block.offset(rightBlockLocation[direction]);
		const back = block.offset(backBlockLocation[direction]);
		const top = block.above();
		const topRight = top.offset(rightBlockLocation[direction]);
		const topBack = top.offset(backBlockLocation[direction]);
		const topRightBack = topRight.offset(backBlockLocation[direction]);
		return [
			{ target: right, part: 2 },
			{ target: right.offset(backBlockLocation[direction]), part: 3 },
			{ target: back, part: 4 },
			{ target: top, part: 5 },
			{ target: topRight, part: 6 },
			{ target: topRightBack, part: 7 },
			{ target: topBack, part: 8 }
		];
	}]
]);
class DropTallWindowInteract {
	constructor() {
		this.onPlayerInteract = this.onPlayerInteract.bind(this);
	}
	getSound(wood) {
		const doorSounds = {
			cherry: { open: 'open.cherry_wood_door', close: 'close.cherry_wood_door' },
			bamboo: { open: 'open.bamboo_wood_door', close: 'close.bamboo_wood_door' },
			nether: { open: 'open.nether_wood_door', close: 'close.nether_wood_door' },
			default: { open: 'open.wooden_door', close: 'close.wooden_door' }
		}
		return doorSounds[wood];
	}
	updateDoorPair(block, newOpenState) {
		const partBlock = block.permutation.getState("mc:block_parts") === 1 ? block.above() : block.below();
		[block, partBlock].forEach(b => b.setPermutation(b.permutation.withState("mc:open", newOpenState)))
	}
	onPlayerInteract(e) {
		const { block, dimension } = e;
		const permutation = block.permutation;
		const direction = permutation.getState("minecraft:cardinal_direction");
		const currentOpenState = permutation.getState("mc:open");
		const newOpenState = !currentOpenState;
		this.updateDoorPair(block, newOpenState);
		const woodType = getBlockTypeVariant(block.typeId);
		const soundKey = currentOpenState ? "close" : "open";
		dimension.playSound(this.getSound(woodType)[soundKey], block.center());
		const sideBlockLoc = permutation.getState("mc:side") ? leftBlockLocation : rightBlockLocation;
		const sideBlock = block.offset(sideBlockLoc[direction]);
		if (sideBlock.hasTag("mc:door")) this.updateDoorPair(sideBlock, newOpenState);
	}
}
class DropTallWindowPlace {
	beforeOnPlayerPlace(e) {
		const { block } = e;
		if (!airBlocks.includes(block.above().typeId)) e.cancel = true;
	}
	onPlace(e) {
		const { block } = e;
		const direction = block.permutation.getState(directionState);
		const blockAbove = block.above();
		const isRightState = block.offset(leftBlockLocation[direction]).hasTag("mc:side_door");
		block.setPermutation(block.permutation.withState("mc:block_parts", 1).withState("mc:side", isRightState));
		blockAbove.setPermutation(BlockPermutation.resolve(block.typeId, { "minecraft:cardinal_direction": direction, "mc:block_parts": 2, "mc:side": isRightState }));
	}
}
class DropLargeBlockPlacer {
	constructor(sizeKey, spawnEntity = undefined) {
		this.blockParts = blockPartGetters.get(sizeKey);
		this.spawnEntity = spawnEntity;
		this.onPlace = this.onPlace.bind(this);
		this.beforeOnPlayerPlace = this.beforeOnPlayerPlace.bind(this);
	}
	setBlockParts(parts, id, direction) {
		parts.forEach(({ target, part }) => {
			const params = { [directionState]: direction, "mc:block_parts": part };
			target.setPermutation(BlockPermutation.resolve(id, params));
		});
	}
	beforeOnPlayerPlace(e) {
		const { block, permutationToPlace } = e;
		const direction = permutationToPlace.getState(directionState);
		const parts = this.blockParts(direction, block);
		if (!parts || !parts.every(({ target }) => airBlocks.includes(target.typeId))) e.cancel = true;
	}
	onPlace(e) {
		const { block } = e;
		const direction = block.permutation.getState(directionState);
		const parts = this.blockParts(direction, block);
		this.setBlockParts(parts, block.typeId, direction);
		block.setPermutation(block.permutation.withState("mc:block_parts", 1));
		if (this.spawnEntity) spawnEntityRotatedByBlock(this.spawnEntity.id, block, direction, this.spawnEntity.offset);
	}
}
class DropBreakLargeBlock {
	onPlayerDestroy(e) {
		if (e.player.matches({ gameMode: GameMode.creative })) return;
		const item = e.player.getComponent("equippable").getEquipment(EquipmentSlot.Mainhand);
		const hasSilkTouch = item?.getComponent(ItemEnchantableComponent.componentId)?.getEnchantment("silk_touch");
		if (item && hasSilkTouch) return;
		e.dimension.spawnItem(new ItemStack(e.destroyedBlockPermutation.type.id, 1), e.block.center());
	}
}
function dropInventory(entity, slots = entity.getComponent("inventory").container.size) {
	const container = entity.getComponent("inventory").container;
	for (let i = 0; i < slots; i++) {
		const item = container.getItem(i);
		if (item) entity.dimension.spawnItem(item, entity.location);
	}
}
system.afterEvents.scriptEventReceive.subscribe(e => {
	const { id, sourceEntity, message } = e;
	if (!sourceEntity || !sourceEntity.matches({ families: [ "drop_return_inv" ] })) return;
	if (id === "drop:return_inventory") {
		dropInventory(sourceEntity);
		sourceEntity.remove();
	} else if (id === "drop:return_inventory_slots") {
		dropInventory(sourceEntity, Number(message));
		sourceEntity.remove();
	}
});
class DropPlaceContainerOld {
	constructor(typeEntity = "drop:container_drop", nameTag = "") {
		this.typeEntity = typeEntity;
		this.nameTag = nameTag;
		this.onPlace = this.onPlace.bind(this);
	}
	onPlace(e) {
		const container = spawnEntityRotatedByBlock(this.typeEntity, e.block, e.block.permutation.getState(directionState));
		container.addEffect("invisibility", 10, { showParticles: false });
		container.triggerEvent(e.block.typeId);
		container.nameTag = this.nameTag;
		e.block.setPermutation(e.block.permutation.withState("drop:placed", true));
	}
}
class DropToilet {
	constructor() {
		this.toggleState = "drop:open_state";
		this.onPlayerInteract = this.onPlayerInteract.bind(this);
	}
	onPlayerInteract(e) {
		const { player, block, dimension } = e;
		if (player.isSneaking) {
			const state = block.permutation.getState(this.toggleState);
			block.setPermutation(block.permutation.withState(this.toggleState, !state));
			dimension.playSound(`block.barrel.${state ? "close": "open"}`, block.center());
		} else sitOn(block, player, "drop:chair_entity_drop");
	}
}
class DropGlobalConfig {
	onUse(e) {
		const { source } = e;
		const config = world.getDynamicProperty("DecoDropConfig");
		const configObject = config ? { ...DecoDropConfig, ...JSON.parse(config) } : DecoDropConfig;
		const configUI = new ModalFormData()
			.title({ translate: 'drop.decodrop.settings' })
			.toggle({ translate: 'drop.decodrop.accelerate_time_when_sleeping' }, configObject.accelerateTime)
			.toggle({ translate: 'drop.decodrop.sleep_only_at_night' }, configObject.onlySleepAtNight);
		configUI.show(source).then(response => {
			if (response.canceled) return;
			const [accelerateTime, onlySleepAtNight] = response.formValues;
			configObject.accelerateTime = accelerateTime;
			configObject.onlySleepAtNight = onlySleepAtNight;
			world.setDynamicProperty("DecoDropConfig", JSON.stringify(configObject));
			initializeSleepConfig();
		});
	}
}
class DropBreakLeftBlock {
	onPlayerDestroy(e) {
		spawnItemSilkTouch(e, `${e.destroyedBlockPermutation.type.id.replace("left", "item")}`)
	}
}
class DropPianoPlace {
	beforeOnPlayerPlace(e) {
		if (!airBlocks.includes(e.block.offset(rightBlockLocation[e.permutationToPlace.getState(directionState)]).type.id)) e.cancel = true;
	}
	onPlace(e) {
		const { block } = e;
		const direction = block.permutation.getState(directionState);
		const rightBlock = block.offset(rightBlockLocation[direction]);
		rightBlock.setPermutation(BlockPermutation.resolve(block.typeId.replace("item", "right"), { [directionState]: direction }));
		block.setPermutation(BlockPermutation.resolve(block.typeId.replace("item", "left"), { [directionState]: direction }));
	}
}
const songMap = {
	"drop:sheet_calm": { soundName: "piano.calm", noteMapKey: PianoCalm },
	"drop:sheet_haggstrom": { soundName: "piano.haggstrom", noteMapKey: PianoHaggstrom },
	"drop:sheet_mice_on_venus": { soundName: "piano.mice_on_venus", noteMapKey: PianoMiceOnVenus },
	"drop:sheet_subwoofer_lullaby": { soundName: "piano.subwoofer_lullaby", noteMapKey: PianoSubwooferLullaby },
	"drop:sheet_sweden": { soundName: "piano.sweden", noteMapKey: PianoSweden },
	"drop:sheet_wet_hands": { soundName: "piano.wet_hands", noteMapKey: PianoWetHands }
};
class DropAddSheetMusic {
	onPlayerInteract(e) {
		const { player, block, dimension } = e;
		const inventory = player.getComponent("inventory").container;
		const item = inventory.getItem(player.selectedSlotIndex);
		const hasSheetMusic = block.permutation.getState("drop:sheet");
		const direction = block.permutation.getState(directionState);
		const pianoOffsets = {
			"drop.piano_left": {
				blockOffset: { north: { x: -1, y: 0, z: 0 }, south: { x: 1, y: 0, z: 0 }, west: { x: 0, y: 0, z: 1 }, east: { x: 0, y: 0, z: -1 } },
				entityOffset: { north: { x: 0, y: 0.5, z: 0.5 }, south: { x: 1, y: 0.5, z: 0.5 }, west: { x: 0.5, y: 0.5, z: 1 }, east: { x: 0.5, y: 0.5, z: 0 } }
			},
			"drop.piano_right": {
				blockOffset: { north: { x: 1, y: 0, z: 0 }, south: { x: -1, y: 0, z: 0 }, west: { x: 0, y: 0, z: -1 }, east: { x: 0, y: 0, z: 1 } },
				entityOffset: { north: { x: 1, y: 0.5, z: 0.5 }, south: { x: 0, y: 0.5, z: 0.5 }, west: { x: 0.5, y: 0.5, z: 0 }, east: { x: 0.5, y: 0.5, z: 1 } }
			}
		};
		const pianoType = block.hasTag("drop.piano_left") ? "drop.piano_left" : "drop.piano_right";
		const offsets = pianoOffsets[pianoType];
		function updateSheetState(state) {
			if (state) dimension.playSound("block.itemframe.add_item", block.center());
			else dimension.playSound("block.itemframe.remove_item", block.center());
			const oppositeBlock = block.offset(offsets.blockOffset[direction]);
			block.setPermutation(block.permutation.withState("drop:sheet", state));
			oppositeBlock.setPermutation(oppositeBlock.permutation.withState("drop:sheet", state));
		}
		function handleSongEnd(pianoEntity, dataSong) {
			player.onScreenDisplay.setActionBar([{ translate: 'drop.piano.song_ends'},{ translate: `${dataSong.song}`}]);
			dataSong.note = 0;
			pianoEntity.setDynamicProperty("drop:song", JSON.stringify(dataSong));
		}
		function playNotes(notesList, location) {
			notesList.forEach(noteInfo => {
				if (noteInfo.sound) dimension.playSound(noteInfo.sound, location, { pitch: noteInfo.pitch, volume: 1 });
			});
		}
		if (item && songMap[item.typeId] && !hasSheetMusic) {
			const PianoUI = new ActionFormData()
				.title({ translate: 'drop.piano.select_playback_mode' })
				.button({ translate: 'drop.piano.auto_mode' })
				.button({ translate: 'drop.piano.note_by_note' });
			PianoUI.show(player).then(response => {
				if (response.canceled) return;
				updateSheetState(true);
				decrementItemInHand(player);
				const pianoEntity = dimension.spawnEntity("drop:piano_entity", {
					x: block.location.x + offsets.entityOffset[direction].x,
					y: block.location.y + offsets.entityOffset[direction].y,
					z: block.location.z + offsets.entityOffset[direction].z
				});
				const songData = {
					song: item.typeId,
					note: 0,
					mode: response.selection === 0 ? "auto" : "manual",
					soundName: songMap[item.typeId].soundName
				};
				if (songData.soundName && response.selection === 0) {
					dimension.playSound(songData.soundName, pianoEntity.location);
					player.onScreenDisplay.setActionBar([{ translate: 'drop.piano.now_playing'},{ translate: `${songData.song}`}]);
				} else player.onScreenDisplay.setActionBar({ translate: `${songData.song}`});
				pianoEntity.setDynamicProperty("drop:song", JSON.stringify(songData));
			});
		} else if (hasSheetMusic) {
			const pianoEntity = dimension.getEntitiesAtBlockLocation(block.location).find(entity => entity.typeId === "drop:piano_entity");
			if (!pianoEntity) {
				updateSheetState(false);
				return;
			}
			const dataSong = JSON.parse(pianoEntity.getDynamicProperty("drop:song"));
			if (!dataSong) {
				updateSheetState(false);
				return;
			}
			if (dataSong.mode === "auto" && dataSong.soundName && player.isSneaking) {
				dimension.runCommand(`stopsound @a ${dataSong.soundName}`);
				addItemOrSpawn(player, new ItemStack(dataSong.song, 1));
				updateSheetState(false);
				pianoEntity.remove();
			} else if (dataSong.mode === "manual") {
				if (player.isSneaking) {
					addItemOrSpawn(player, new ItemStack(dataSong.song, 1));
					updateSheetState(false);
					pianoEntity.remove();
				} else if (songMap[dataSong.song]) {
					const PianoSong = songMap[dataSong.song].noteMapKey;
					const notesList = PianoSong[dataSong.note];
					if (Array.isArray(notesList) && notesList.length > 0) {
						playNotes(notesList, pianoEntity.location);
						dataSong.note++;
						if (dataSong.note >= PianoSong.length) {
							handleSongEnd(pianoEntity, dataSong);
						} else {
							pianoEntity.setDynamicProperty("drop:song", JSON.stringify(dataSong));
							player.onScreenDisplay.setActionBar(`${dataSong.note} / ${PianoSong.length}`)
						}
					}
				}
			}
		}
	}
	onPlayerDestroy(e) {
		const { dimension, block } = e;
		const pianoEntity = dimension.getEntitiesAtBlockLocation(block.location).find(entity => entity.typeId === "drop:piano_entity");
		if (!pianoEntity) return;
		const dataSong = JSON.parse(pianoEntity.getDynamicProperty("drop:song"));
		if (!dataSong) return;
		dimension.runCommand(`stopsound @a ${dataSong.soundName}`);
		dimension.spawnItem(new ItemStack(dataSong.song, 1), block.center());
	}
}
class DropDrinkMilk {
	onConsume(e) {
		const effects = ["nausea", "poison", "fatal_poison", "slowness", "weakness", "wither", "mining_fatigue", "levitation", "hunger", "darkness", "blindness", "bad_omen", "infested", "oozing", "weaving", "wind_charged"];
		effects.forEach(effect => e.source.removeEffect(effect));
	}
}
class DropDrinkMate {
	onConsume(e) {
		e.source.addEffect("regeneration", 100, { "amplifier": 0 });
	}
}
class DropCopperTeapotConsume {
	onPlayerInteract(e) {
		const { player, block, dimension } = e;
		const teapotEntity = dimension.getEntitiesAtBlockLocation(block.location).find(entity => entity.typeId === "drop:copper_teapot");
		if (!teapotEntity) {
			block.setPermutation(block.permutation.withState("drop:food", "empty"));
			return;
		}
		const item = player.getComponent("inventory").container.getItem(player.selectedSlotIndex);
		if (item?.typeId.includes("leaves") && block.permutation.getState("drop:food") === "water") {
			teapotEntity.setProperty("drop:icon", 2);
			block.setPermutation(block.permutation.withState("drop:food", "mate"));
			dimension.playSound("block.itemframe.add_item", block.center());
			decrementItemInHand(player);
			return;
		}
		if (item?.typeId !== "drop:cup_empty_item") return;
		const timer = teapotEntity.getProperty("drop:timer");
		if (timer >= 15) {
			const newItemType = teapotEntity.getProperty("drop:icon");
			const itemConvertsTo = ["drop:cup_water_item", "drop:cup_milk_item", "drop:cup_mate_item"];
			dimension.playSound("bottle.dragonbreath", block.center());
			addItemOrSpawn(player, new ItemStack(itemConvertsTo[newItemType], 1));
			decrementItemInHand(player);
			const quantity = teapotEntity.getDynamicProperty("drop:quantity") || 0;
			if (quantity > 0) {
				teapotEntity.setDynamicProperty("drop:quantity", quantity - 1);
			} else {
				block.setPermutation(block.permutation.withState("drop:food", "empty"));
				teapotEntity.remove();
			}
		}
	}
}
class DropBreakFlower {
	onPlayerDestroy(e) {
		spawnItemSilkTouch(e, e.destroyedBlockPermutation.type.id.replace("drop:flower_","minecraft:"));
	}
}
const allowedDirt = ["minecraft:grass_block", "minecraft:grass_path", "minecraft:dirt", "minecraft:coarse_dirt", "minecraft: dirt_with_roots", "minecraft:farmland", "minecraft:podzol"];
class DropLargePotAddDirt {
	onPlayerInteract(e) {
		const { player, block, dimension } = e;
		const item = player.getComponent("equippable").getEquipment(EquipmentSlot.Mainhand);
		if (allowedDirt.includes(item?.typeId)) {
			block.setType(`${block.typeId}_dirt`);
			dimension.playSound("block.composter.fill", block.center());
			decrementItemInHand(player);
		} else player.onScreenDisplay.setActionBar({ translate: 'drop.actionbar.add_dirt' });
	}
}
class DropLargePotPlant {
	onPlayerInteract(e) {
		const { player, block, dimension } = e;
		const item = player.getComponent("inventory").container.getItem(player.selectedSlotIndex);
		const allowedFlowers = ["minecraft:dandelion", "minecraft:poppy", "minecraft:blue_orchid", "minecraft:allium", "minecraft:azure_bluet", "minecraft:red_tulip", "minecraft:orange_tulip", "minecraft:white_tulip", "minecraft:pink_tulip", "minecraft:oxeye_daisy", "minecraft:cornflower", "minecraft:lily_of_the_valley", "minecraft:torchflower", "minecraft:wither_rose", "minecraft:closed_eyeblossom", "minecraft:open_eyeblossom"];
		if (allowedFlowers.includes(item?.typeId) && airBlocks.includes(block.above().typeId)) {
			block.above().setType(item.typeId.replace("minecraft:", "drop:flower_"));
			dimension.playSound("dig.azalea_leaves", block.center());
			decrementItemInHand(player);
		}
	}
}
class DropSitOnChair2 {
	onPlayerInteract(e) {
		handleSitOnFurniture(e, "drop:chair_entity_drop", 0.4);
	}
}
const getBlockState = (rot) => {
	const normalized = rot + 180;
	const sector = Math.floor((normalized + 22.5) / 45) % 8;
	const directions = ["south", "south", "west", "west", "north", "north", "east", "east"];
	const rotates = [false, true, false, true, false, true, false, true];
	return { direction: directions[sector], rotate: rotates[sector] };
};
class DropRotateBy45 {
	beforeOnPlayerPlace(e) {
		const { player } = e;
		const yRot = player.getRotation().y;
		const { direction, rotate } = getBlockState(yRot);
		e.permutationToPlace = e.permutationToPlace.withState("minecraft:cardinal_direction", direction).withState("drop:rotate", rotate);
	}
}
class DropCeramicStation {
	onPlayerInteract(e) {
		const { player, block, dimension } = e;
		const item = player.getComponent("inventory").container.getItem(player.selectedSlotIndex);
		if (item?.typeId !== "minecraft:clay") {
			player.onScreenDisplay.setActionBar({ translate: 'drop.ceramic_station.use_clay_blocks' });
			return;
		}
		const CeramicUI = new ActionFormData()
			.title("DecoDrop Ceramics Station")
			.button("", "textures/ui/drop/clay_teapot")
			.button("", "textures/ui/drop/clay_pot")
			.button("", "textures/ui/drop/clay_beaker")
			.button("", "textures/ui/drop/clay_large_pot")
			.button("", "textures/ui/drop/clay_liquid_container")
			.button("", "textures/ui/drop/clay_toilet");
		CeramicUI.show(player).then(response => {
			if (response.canceled) return;
			const newItem = ["drop:clay_teapot", "drop:clay_pot", "drop:clay_beaker", "drop:clay_large_pot", "drop:water_pot_clay", "drop:toilet_clay"][response.selection];
			const CeramicUIquantity = new ModalFormData()
				.title({ translate: 'drop.ceramic_station.select_item_quantity' })
				.slider({ translate: 'drop.ceramic_station.item_quantity' }, 1, item.amount, 1, 1);
			CeramicUIquantity.show(player).then(response => {
				if (response.canceled) return;
				const [ quantity ] = response.formValues;
				decrementItemInHand(player, false, quantity);
				dimension.spawnItem(new ItemStack(newItem, quantity), block.center());
			});
		});
	}
}
class DropBreakTable2x2 {
	onPlayerDestroy(e) {
		spawnItemSilkTouch(e, `${e.destroyedBlockPermutation.type.id.replace("_00","").replace("_01","").replace("_10","").replace("_11","")}`);
	}
}
class DropPlaceTable2x2 {
	beforeOnPlayerPlace(e) {
		const { player, block, dimension, permutationToPlace } = e;
		const direction = getDirectionByPlayer(player);
		const rightBlock = block.offset(rightBlockLocation[direction]);
		const backBlock = block.offset(backBlockLocation[direction]);
		const rightBackBlock = rightBlock.offset(backBlockLocation[direction]);
		const blockId = permutationToPlace.type.id;
		if ([rightBlock.typeId, backBlock.typeId, rightBackBlock.typeId].every(typeId => airBlocks.includes(typeId))) {
			const permutationsMap = {
				north: ["_00", "_01", "_11", "_10"],
				south: ["_10", "_11", "_01", "_00"],
				west: ["_01", "_10", "_00", "_11"],
				east: ["_11", "_00", "_10", "_01"]
			};
			const permutations = permutationsMap[direction];
			[block, rightBlock, backBlock, rightBackBlock].forEach((targetBlock, index) => {
				targetBlock.setPermutation(BlockPermutation.resolve(`${blockId}${permutations[index]}`));
			});
			decrementItemInHand(player);
			const sound = blockId.includes("bamboo") ? "place.bamboo_wood" : blockId.includes("cherry") ? "place.cherry_wood" : blockId.includes("crimson") || blockId.includes("warped") ? "place.nether_wood" : "dig.wood";
			dimension.playSound(sound, block.center());
		}
		e.cancel = true;
	}
}
const SLEEP_ANIM = {
	"drop:sleep_bed": "animation.drop.player.sleep_old",
	"drop:bed_0_entity": "animation.drop.player.sleep_0",
	"drop:bed_1": "animation.drop.player.sleep_0"
}
world.afterEvents.dataDrivenEntityTrigger.subscribe(({ entity }) => {
	try {
		if (!entity) return;
		const player = entity?.getComponent("rideable")?.getRiders()?.[0];
		if (!player) return;
		player.playAnimation(SLEEP_ANIM[entity?.typeId], { stopExpression: "!q.is_riding" } );
	} catch (error) {}
}, {
	entityTypes: [ "drop:bed_0_entity", "drop:bed_1", "drop:sleep_bed" ],
	eventTypes: [ "drop:set_player_anim" ]
});

let accelerateTime = false;
let onlySleepAtNight = true;

function initializeSleepConfig() {
	const config = world.getDynamicProperty("DecoDropConfig");
	const configObject = config ? { ...DecoDropConfig, ...JSON.parse(config) } : DecoDropConfig;
	accelerateTime = configObject.accelerateTime;
	onlySleepAtNight = configObject.onlySleepAtNight;
}
initializeSleepConfig();

const overworld = world.getDimension("overworld");
let sleepIntervalId = null;
let addTimeLoop = null;
let sleepingPlayers = [];
let isFading = false;
let targetTime = null;
let requiredSleepingPlayers = false;

const isNightTime = () => {
	const time = world.getTimeOfDay();
	return time >= 13000 && time < 23000;
};
const stopTimeAcceleration = () => {
	if (addTimeLoop) {
		system.clearRun(addTimeLoop);
		targetTime = null;
		addTimeLoop = null;
		stopSleepInterval();
	}
};
const showWakeUpEffects = () => {
	if (isFading) return;
	isFading = true;
    const allPlayers = world.getAllPlayers();
	sleepingPlayers.forEach(player => {
		player.camera.fade({
			fadeTime: { fadeInTime: 2.0, holdTime: 0.0, fadeOutTime: 1.0 },
			fadeColor: { red: 0.0, green: 0.0, blue: 0.0 }
		});
	});
	system.runTimeout(() => {
		if (!requiredSleepingPlayers) return;
		overworld.setWeather("Clear");
		world.setTimeOfDay(targetTime || 0);
		const adjustedTime = ((targetTime || 0) + 6000) % 24000;
		const hours = Math.floor(adjustedTime / 1000);
		const minutes = Math.floor(((adjustedTime % 1000) / 1000) * 60);
		allPlayers.forEach(player => {
			player.onScreenDisplay.setActionBar(`${hours.toString().padStart(2, '0')}:${minutes.toString().padStart(2, '0')}`);
		});
		isFading = false;
		targetTime = null;
		stopSleepInterval();
	}, 40);
};
const accelerateTimeCycle = () => {
	if (addTimeLoop) return;
	addTimeLoop = system.runInterval(() => {
		const currentTime = world.getTimeOfDay();
		const newTime = (currentTime + 30) % 24000;
		world.setTimeOfDay(newTime);
		const timeDifference = (targetTime - currentTime + 24000) % 24000;
		if (!requiredSleepingPlayers) return stopTimeAcceleration();
		if (timeDifference <= 40) {
			overworld.setWeather("Clear");
			world.setTimeOfDay(targetTime);
			stopTimeAcceleration();
			targetTime = null;
		}
	}, 1);
};
function startSleepInterval() {
	if (sleepIntervalId !== null) return;
	sleepIntervalId = system.runInterval(() => {
		if (isFading) return;
		const allPlayers = world.getAllPlayers();
		sleepingPlayers = allPlayers.filter(player => 
			player.getComponent("minecraft:riding")?.entityRidingOn?.matches({ families: [ "drop_sleep_anim" ] })
		);
		if (sleepingPlayers.length === 0) return stopSleepInterval();
		const requiredCount = Math.ceil(world.gameRules.playersSleepingPercentage / 100 * allPlayers.length);
		requiredSleepingPlayers = sleepingPlayers.length >= requiredCount;
		if (!requiredSleepingPlayers) {
			sleepingPlayers.forEach(player => {
				player.onScreenDisplay.setActionBar({ translate: "drop.actionbar.sleeping_players", with: [`${sleepingPlayers.length}/${requiredCount}`] });
			});
			return;
		} else if (targetTime !== null) {
			if (accelerateTime) accelerateTimeCycle();
			else showWakeUpEffects();
		}
	}, 20);
}
function stopSleepInterval() {
	if (sleepIntervalId) {
		system.clearRun(sleepIntervalId);
		sleepIntervalId = null;
	}
}
class DropSignIcon {
	beforeOnPlayerPlace(e) {
		const { face, permutationToPlace } = e;
		if (face === Direction.Up || face === Direction.Down) return;
		e.permutationToPlace = permutationToPlace.withState(directionState, face.toLowerCase());
	}
	onPlayerInteract(e) {
		const { player, block, dimension } = e;
		const icon = block.permutation.getState("drop:icon");
		const newIcon = !player.isSneaking ? (icon + 1) % 10 : (icon - 1 + 10) % 10;
		block.setPermutation(block.permutation.withState("drop:icon", newIcon));
		dimension.playSound("block.itemframe.add_item", block.center());
	}
}
function PlayersSleep(player) {
	if (!sleepIntervalId) startSleepInterval();
	if (onlySleepAtNight) {
		targetTime = 0;
		return;
	}
	const SleepUI = new ActionFormData()
		.title({ translate: 'drop.bed.select_time_of_day' })
		.button({ translate: 'hostOption.time.sunrise' }, "textures/ui/drop/time_sunrise")     // 23000
		.button({ translate: 'hostOption.time.day' }, "textures/ui/drop/time_day")             // 1000
		.button({ translate: 'hostOption.time.noon' }, "textures/ui/drop/time_noon")           // 6000
		.button({ translate: 'hostOption.time.sunset' }, "textures/ui/drop/time_sunset")       // 12000
		.button({ translate: 'hostOption.time.night' }, "textures/ui/drop/time_night")         // 13000
		.button({ translate: 'hostOption.time.midnight' }, "textures/ui/drop/time_midnight");  // 18000
	SleepUI.show(player).then(response => {
		if (response.canceled) return;
		const selectedTime = [23000, 1000, 6000, 12000, 13000, 18000][response.selection];
		targetTime = selectedTime;
	});
}
function checkSpawnPoint(player, playerSpawnpoint, newSpawnpoint) {
	const areSpawnPointsEqual = playerSpawnpoint?.dimension.id === newSpawnpoint.dimension.id && playerSpawnpoint?.x === newSpawnpoint.x && playerSpawnpoint?.y === newSpawnpoint.y && playerSpawnpoint?.z === newSpawnpoint.z;
	if (!areSpawnPointsEqual) {
		player.setSpawnPoint(newSpawnpoint);
		player.sendMessage({ rawtext: [{ text: '§i' }, { translate: 'tile.bed.respawnSet' }] });
	}
}
function checkDimensionExplosion(dimension) {
	if (dimension.id !== "minecraft:overworld") {
		dimension.createExplosion(center, 5, {breaksBlocks: true, causesFire: true});
		return true;
	} return false;
}
class DropSleepInBed {
	constructor() {
		this.onPlayerInteract = this.onPlayerInteract.bind(this);
	}
	sleepPlayer(player, bed) {
		bed.getComponent("rideable").addRider(player);
		if (!targetTime) PlayersSleep(player);
	}
	onPlayerInteract(e) {
		const { player, block, dimension } = e;
		const bed = dimension.getEntitiesAtBlockLocation(block.location).find((entity) => entity.matches({ families: [ "drop_bed" ] }));
		if (!bed) return;
		if (checkDimensionExplosion(dimension)) return;
		if (bed.getProperty("drop:has_rider")) return;
		checkSpawnPoint(player, player.getSpawnPoint(), { ...bed.location, dimension: player.dimension });
		if (onlySleepAtNight) {
			if (isNightTime()) this.sleepPlayer(player, bed);
			else player.sendMessage({ rawtext: [{ text: '§i' }, { translate: 'tile.bed.noSleep' }] });
		} else this.sleepPlayer(player, bed);
	}
}
class DropBedSleep {
	onPlayerInteract(e) {
		const { player, block, dimension } = e;
		if (dimension.getEntitiesAtBlockLocation(block.location).some(entity => entity.typeId === "drop:sleep_bed")) return;
		let center = block.center();
		const direction = block.permutation.getState("minecraft:cardinal_direction");
		if (block.hasTag("drop.bed_01")) {
			center = block.offset(leftBlockLocation[direction]).center();
		} else if (block.hasTag("drop.bed_10")) {
			center = block.offset(frontBlockLocation[direction]).center();
		} else if (block.hasTag("drop.bed_11")) {
			center = block.offset(leftBlockLocation[direction]).offset(frontBlockLocation[direction]).center();
		}
		switch (direction) {
			case "north": center.z += 0.5; center.x -= 0.5; break;
			case "south": center.z -= 0.5; center.x += 0.5; break;
			case "west":  center.x += 0.5; center.z += 0.5; break;
			case "east":  center.x -= 0.5; center.z -= 0.5; break;
		}
		if (checkDimensionExplosion(dimension)) return;
		const playerSpawnpoint = player.getSpawnPoint();
		const newSpawnpoint = { dimension: dimension, x: center.x, y: center.y + 0.5, z: center.z };
		checkSpawnPoint(player, playerSpawnpoint, newSpawnpoint);
		function placeBedAndSleep(player, dimension, center, direction) {
			const bedEntity = dimension.spawnEntity("drop:sleep_bed", center);
			bedEntity.setRotation({ x: 0, y: { north: 180, south: 0, west: 90, east: -90 }[direction] });
			bedEntity.getComponent("rideable").addRider(player);
			if (!targetTime) PlayersSleep(player);
		}
		if (onlySleepAtNight) {
			if (isNightTime()) placeBedAndSleep(player, dimension, center, direction);
			else player.sendMessage({ rawtext: [{ text: '§i' }, { translate: 'tile.bed.noSleep' }] });
		} else {
			placeBedAndSleep(player, dimension, center, direction);
		}
	}
}
class DropBreakBed {
	onPlayerDestroy(e) {
		const color = extractColor(e.destroyedBlockPermutation.type.id);
		spawnItemSilkTouch(e, `drop:bed_2x2_${color}`);
	}
}
class DropPlaceBed {
	beforeOnPlayerPlace(e) {
		const { player, block, dimension, permutationToPlace } = e;
		const direction = getDirectionByPlayer(player);
		const rightBlock = block.offset(rightBlockLocation[direction]);
		const backBlock = block.offset(backBlockLocation[direction]);
		const rightBackBlock = rightBlock.offset(backBlockLocation[direction]);
		if ([rightBlock.typeId, backBlock.typeId, rightBackBlock.typeId].every(typeId => airBlocks.includes(typeId))) {
			const color = extractColor(permutationToPlace.type.id);
			block.setPermutation(BlockPermutation.resolve(`drop:bed_2x2_${color}_00`, {"minecraft:cardinal_direction": direction}));
			rightBlock.setPermutation(BlockPermutation.resolve(`drop:bed_2x2_${color}_01`, {"minecraft:cardinal_direction": direction}));
			backBlock.setPermutation(BlockPermutation.resolve(`drop:bed_2x2_${color}_10`, {"minecraft:cardinal_direction": direction}));
			rightBackBlock.setPermutation(BlockPermutation.resolve(`drop:bed_2x2_${color}_11`, {"minecraft:cardinal_direction": direction}));
			decrementItemInHand(player);
			dimension.playSound("dig.wood", block.center());
		}
		e.cancel = true;
	}
}
class DropItemBreakBlock {
	onMineBlock(e) {
		itemApplyDamage(e.source, e.itemStack);
	}
}
class DropConsumeGoldenCake {
	onConsume(e) {
		const { source } = e;
		source.addEffect("regeneration", 100, { "amplifier": 1 });
		source.addEffect("absorption", 2400, { "amplifier": 0 });
	}
}
class DropGetCookingRecipe{
	onUse(e) {
		const { source } = e;
		source.runCommand("structure load drop.cooking_recipes ~~~ 0_degrees");
		decrementItemInHand(source);
	}
}
class DropCandleTable {
	onPlayerInteract(e) {
		const { player, block, dimension } = e;
		const item = player.getComponent("inventory").container.getItem(player.selectedSlotIndex);
		if (item?.typeId === "minecraft:flint_and_steel") {
			const permutation = block.permutation;
			if (!permutation.getState("drop:turn_on")) {
				block.setPermutation(permutation.withState("drop:turn_on", true));
				dimension.playSound("fire.ignite", block.location);
				itemApplyDamage(player, item);
			}
		}
	}
}
class DropSitOnChair {
	onPlayerInteract(e) {
		handleSitOld(e, "drop:chair_entity_drop", 0.56);
	};
}
class DropSitWoodenStool {
	onPlayerInteract(e) {
		handleSitOld(e, "drop:chair_entity_drop", 0.5);
	};
}
function updateConnections(block, tag) {
	let permutation = block.permutation;
	const neighbors = {"mc:n": block.north(),"mc:s": block.south(),"mc:w": block.west(),"mc:e": block.east()};
	for (const state in neighbors) {
		const neighbor = neighbors[state];
		permutation = permutation.withState(state, neighbor && (solidBlock.has(neighbor.typeId) || neighbor.hasTag("mc:solid_block") || neighbor.hasTag(tag)));
	}
	block.setPermutation(permutation);
}
const connectionTags = new Set(["mc:bars", "mc:fence", "mc:wall", "mc:hedge"]);
function checkAdjacentBlocksForConnection(block) {
	const neighbors = [block.north(), block.south(), block.west(), block.east()];
	for (const neighbor of neighbors) {
		if (!neighbor || !neighbor.hasTag("mc:can_connect")) continue;
		for (const tag of connectionTags) {
			if (neighbor.hasTag(tag)) updateConnections(neighbor, tag);
		}
	}
}
world.afterEvents.playerPlaceBlock.subscribe((e) => {
	checkAdjacentBlocksForConnection(e.block);
});
world.afterEvents.playerBreakBlock.subscribe((e) => {
	checkAdjacentBlocksForConnection(e.block);
});
class DropUpdateBlockConnections {
	constructor(blockType) {
		this.blockType = blockType;
		this.onPlace = this.onPlace.bind(this);
	}
	onPlace(e) {
		const block = e.block;
		updateConnections(block, this.blockType);
		const blockAbove = block.above();
		if (!blockAbove.isAir) return;
		blockAbove.setType("drop:fence_top");
	}
}
class DropClayBeaker {
	onPlayerInteract(e) {
		const { player, block, dimension } = e;
		const quantity = block.permutation.getState("drop:quantity");
		const item = player.getComponent("equippable").getEquipment(EquipmentSlot.Mainhand);
		const updateBlock = (newQuantity) => {
			block.setPermutation(block.permutation.withState("drop:quantity", newQuantity));
			dimension.playSound("break.decorated_pot", block.location);
		};
		if (!item || block.typeId !== item.typeId || quantity === 3) {
			if (quantity > 1 && quantity <= 3) {
				updateBlock(quantity - 1);
				addItemOrSpawn(player, new ItemStack(block.typeId, 1));
			}
		} else if (quantity < 3) {
			updateBlock(quantity + 1);
			decrementItemInHand(player);
		}
	}
}
const DRY_CLAY_MAP = {
	"drop:clay_beaker": "drop:terracotta_clay_beaker",
	"drop:clay_teapot": "drop:terracotta_teapot",
	"drop:clay_pot": "drop:terracotta_pot",
	"drop:clay_large_pot": "drop:terracotta_large_pot",
	"drop:water_pot_clay": "drop:water_pot_terracotta",
	"drop:toilet_clay": "drop:toilet_terracotta"
};
class DropDryClay {
	onRandomTick(e) {
		const { block, dimension } = e;
		if (block.below().hasTag("mc:heat_source")) {
			const blockId = block.typeId;
			const newBlockType = DRY_CLAY_MAP[blockId];
			if (!newBlockType) return;
			const direction = block.permutation.getState(directionState);
			block.setPermutation(BlockPermutation.resolve(DRY_CLAY_MAP[blockId], direction ? { [directionState]: direction } : undefined));
			dimension.playSound("break.decorated_pot", block.center());
		}
	}
}
class DropLamp {
	onPlayerInteract(e) {
		toggleBlockState(e, "drop:state_on", "effect.lamp_on", "effect.lamp_off");
	}
}
class DropVase {
	onPlayerInteract(e) {
		const { player, block, dimension } = e;
		const inventory = player.getComponent("inventory").container;
		const item = inventory.getItem(player.selectedSlotIndex);
		const fluidLevel = block.permutation.getState("drop:fill");
		const fluidType = block.permutation.getState("drop:fluid");
		const fluidActions = {
			"minecraft:water_bucket": { fill: 11, type: "water", sound: "bucket.empty_water", replaceItem: "minecraft:bucket" },
			"minecraft:potion": { fill: 1, type: "water", sound: "bottle.dragonbreath", replaceItem: "minecraft:glass_bottle" },
			"minecraft:lava_bucket": { fill: 11, type: "lava", sound: "bucket.empty_lava", replaceItem: "minecraft:bucket" },
			"minecraft:honey_bottle": { fill: 1, type: "honey", sound: "bottle.dragonbreath", replaceItem: "minecraft:glass_bottle" },
			"minecraft:slime_ball": { fill: 1, type: "slime", sound: "step.slime", replaceItem: "minecraft:slime_ball" },
			"minecraft:milk_bucket": { fill: 11, type: "milk", sound: "bucket.empty_water", replaceItem: "minecraft:bucket" }
		};
		const bucketActions = {
			"water": { itemType: "minecraft:water_bucket", sound: "bucket.fill_water" },
			"lava": { itemType: "minecraft:lava_bucket", sound: "bucket.fill_lava" },
			"milk": { itemType: "minecraft:milk_bucket", sound: "bucket.fill_water" }
		};
		if (fluidLevel === 0 && fluidType === "none" && fluidActions[item?.typeId]) {
			const action = fluidActions[item.typeId];
			block.setPermutation(block.permutation.withState("drop:fill", action.fill).withState("drop:fluid", action.type));
			dimension.playSound(action.sound, block.location);
			if (item.typeId.includes("bucket")) {
				addOrReplaceItem(player, action.replaceItem, 1);
			} else {
				decrementItemInHand(player);
				if (item.typeId !== "minecraft:slime_ball") {
					addItemOrSpawn(player, new ItemStack(action.replaceItem, 1));
				}
			}
		}
		if (fluidLevel > 0 && fluidLevel < 11 && item) {
			const action = fluidActions[item.typeId];
			if (action && action.type === fluidType) {
				const newFluidLevel = Math.min(fluidLevel + (action.fill === 11 ? 11 : 1), 11);
				block.setPermutation(block.permutation.withState("drop:fill", newFluidLevel));
				dimension.playSound(action.sound, block.center());
				if (item.typeId.includes("bucket")) {
					addOrReplaceItem(player, action.replaceItem, 1);
				} else {
					decrementItemInHand(player);
					if (item.typeId !== "minecraft:slime_ball") {
						addItemOrSpawn(player, new ItemStack(action.replaceItem, 1));
					}
				}
			}
		}
		if (item?.typeId === "minecraft:glass_bottle" && fluidType !== "none") {
			const bottleTypeMap = {
				"water": { itemType: "minecraft:potion", fillSound: "bucket.fill_water", fillLevel: 1 },
				"honey": { itemType: "minecraft:honey_bottle", fillSound: "random.drink_honey", fillLevel: 1 }
			};
			const bottleAction = bottleTypeMap[fluidType];
			if (bottleAction && fluidLevel > 0) {
				const newFluidLevel = fluidLevel - bottleAction.fillLevel;
				block.setPermutation(block.permutation.withState("drop:fill", newFluidLevel));
				decrementItemInHand(player);
				if (item.typeId !== "minecraft:slime_ball") {
					addItemOrSpawn(player, new ItemStack(bottleAction.itemType, 1));
				}
				dimension.playSound("bottle.dragonbreath", block.location);
				dimension.playSound(bottleAction.fillSound, block.location);
				if (newFluidLevel === 0) {
					block.setPermutation(block.permutation.withState("drop:fill", 0).withState("drop:fluid", "none"));
				}
			}
		}
		if (item?.typeId === "minecraft:bucket" && fluidLevel === 11 && bucketActions[fluidType]) {
			const bucketAction = bucketActions[fluidType];
			addOrReplaceItem(player, bucketAction.itemType, 1);
			block.setPermutation(block.permutation.withState("drop:fill", 0).withState("drop:fluid", "none"));
			dimension.playSound(bucketAction.sound, block.location);
		}
		if (fluidType === "slime" && item?.typeId !== "minecraft:slime_ball") {
			dimension.playSound("step.slime", block.location);
			addItemOrSpawn(player, new ItemStack("minecraft:slime_ball", 1));
			if (fluidLevel > 1) block.setPermutation(block.permutation.withState("drop:fill", fluidLevel - 1));
			else block.setPermutation(block.permutation.withState("drop:fill", 0).withState("drop:fluid", "none"));
		}
	}
	onPlayerDestroy(e) {
		const { player, block, dimension, destroyedBlockPermutation } = e;
		if (isCreative(player)) return;
		const fluidLevel = destroyedBlockPermutation.getState("drop:fill");
		const fluidType = destroyedBlockPermutation.getState("drop:fluid");
		if (fluidLevel > 0) {
			if (fluidType === "slime") {
				dimension.spawnItem(new ItemStack("minecraft:slime_ball", fluidLevel), block.center());
			}
		}
	}
}
class DropPetBed {
	onPlayerInteract(e) {
		sitPet(e, ["minecraft:wolf", "minecraft:cat"], 0.25, 7);
	}
}
class DropBirdBox {
	onPlayerInteract(e) {
		sitPet(e, ["minecraft:parrot", "minecraft:chicken"], 0.2, 7);
	}
}
class DropHangingPot {
	onPlayerInteract(e) {
		const { player, block, dimension } = e;
		const inventory = player.getComponent("inventory").container;
		const item = inventory.getItem(player.selectedSlotIndex);
		const flowerState = block.permutation.getState("drop:flower");
		const allowedItems = ["minecraft:bone_meal", "minecraft:shears"];
		if (!allowedItems.includes(item?.typeId)) {
			player.onScreenDisplay.setActionBar({ translate: 'drop.hanging_pot.use_bone_meal' });
		}
		if (item?.typeId === "minecraft:bone_meal") {
			block.setPermutation(block.permutation.withState("drop:flower", Math.floor(Math.random() * 3) + 1));
			dimension.playSound("item.bone_meal.use", block.location);
			decrementItemInHand(player);
		} else if (flowerState > 0 && item?.typeId === "minecraft:shears") {
			block.setPermutation(block.permutation.withState("drop:flower", 0));
			dimension.playSound("mob.sheep.shear", block.location);
			if (!isCreative(player)) itemApplyDamage(player, item);
		}
	}
}
class DropFlattenDough {
	onPlayerInteract(e) {
		const { player, block, dimension } = e;
		const item = player.getComponent("equippable").getEquipment(EquipmentSlot.Mainhand);
		const isTool = item?.hasTag("mc:is_flattening_tool");
		const prob = isTool ? 1 : 0.1;
		if (isTool) itemApplyDamage(player, item);
		if (Math.random() <= prob) {
			const flattenState = block.permutation.getState("drop:flatten_dough");
			block.setPermutation(block.permutation.withState("drop:flatten_dough", flattenState + 1));
		}
		dimension.playSound("dig.honey_block", block.location);
	}
}
const addIngredientsMap = {
	"minecraft:apple": "drop:apple_pie_dough",
	"minecraft:cocoa_beans": "drop:chocolate_pie_dough",
	"minecraft:pumpkin": "drop:pumpkin_pie_dough",
	"drop:tomato": "drop:beetroot_dough",
	"drop:dough_item": "drop:cake_dough"
};
class DropAddIngredients {
	onPlayerInteract(e) {
		const { player, block, dimension } = e;
		const item = player.getComponent("equippable").getEquipment(EquipmentSlot.Mainhand);
		if (item?.typeId in addIngredientsMap) {
			const newBlockType = addIngredientsMap[item.typeId];
			block.setType(newBlockType);
			dimension.playSound("block.itemframe.add_item", block.center());
			decrementItemInHand(player);
		}
		if (item?.hasTag("mc:is_knife")) {
			dimension.playSound("block.itemframe.remove_item", block.center());
			dimension.spawnItem(new ItemStack("drop:spaghetti", 3), block.bottomCenter());
			block.setType("minecraft:air");
			itemApplyDamage(player, item);
		}
	}	
}
function UpdateSofa(block) {
	const permutation = block.permutation;
	const direction = permutation.getState("minecraft:cardinal_direction");
	const tag = `${block.typeId}_${direction}`;
	const rightBlock = block.offset(leftBlockLocation[direction]);
	const leftBlock = block.offset(rightBlockLocation[direction]);
	block.setPermutation(permutation.withState("drop:right", rightBlock.hasTag(tag)).withState("drop:left", leftBlock.hasTag(tag)));
}
function TestSofa(block, permutation) {
	let id = block.typeId;
	let direction = permutation.getState("minecraft:cardinal_direction");
	if (id !== "minecraft:air") UpdateSofa(block);
	else id = permutation.type.id;
	const tag = `${id}_${direction}`;
	const rightBlock = block.offset(leftBlockLocation[direction]);
	const leftBlock = block.offset(rightBlockLocation[direction]);
	[rightBlock, leftBlock].forEach(neighbor => { if (neighbor.hasTag(tag)) UpdateSofa(neighbor)});
}
class DropSofa {
	onPlayerInteract(e) {
		handleSitOld(e, "drop:chair_entity_drop", 0.4);
	};
	onPlace(e) {
		TestSofa(e.block, e.block.permutation);
	}
	onPlayerDestroy(e) {
		TestSofa(e.block, e.destroyedBlockPermutation);
	}
}
class DropPizzaAddIngredients {
	onPlayerInteract(e) {
		const { player, block, dimension } = e;
		const inventory = player.getComponent("inventory").container;
		const item = inventory.getItem(player.selectedSlotIndex);
		const states = block.permutation.getState("drop:states");
		if (item?.typeId === "drop:cheese_item" && states === "none") {
			dimension.playSound("block.itemframe.add_item", block.location);
			block.setPermutation(block.permutation.withState("drop:states", "cheese"));
			decrementItemInHand(player);
		}
		if (item?.typeId === "minecraft:porkchop" && states === "cheese") {
			dimension.playSound("block.itemframe.add_item", block.center());
			block.setPermutation(BlockPermutation.resolve("drop:pizza_dough"));
			decrementItemInHand(player);
		}
	}
}
class DropEatPie {
	onPlayerInteract(e) {
		const { player, block, dimension } = e;
		const item = player.getComponent("equippable").getEquipment(EquipmentSlot.Mainhand);
		const amount = block.permutation.getState("drop:pieces");
		if (item?.typeId === `${block.typeId}_item` && amount === 0) {
			const direction = block.permutation.getState(directionState);
			block.setPermutation(BlockPermutation.resolve(block.typeId.replace("block", "stack"), { [directionState]: direction }));
			dimension.playSound("block.itemframe.add_item", block.center());
			decrementItemInHand(player);
			return;
		}
		removePiecesOrBlock(block, dimension);
		dimension.playSound("random.eat", block.center());
		player.addEffect("saturation", 1, { "amplifier": 3, "showParticles": false });
	}
}
const cakeItemMap = {
	"drop:cake_base": "drop:cake_slice_base",
	"drop:cake_chocolate": "drop:cake_slice_chocolate",
	"drop:cake_gold": "drop:cake_slice_gold",
	"drop:cheese_block": "drop:cheese_item",
	"drop:pizza_block": "drop:pizza_slice_item"
}
function removePiecesOrBlock(block) {
	const pieces = block.permutation.getState("drop:pieces");
	if (pieces < 3) block.setPermutation(block.permutation.withState("drop:pieces", pieces + 1));
	else block.setType("minecraft:air");
}
class DropCutBlock {
	onPlayerInteract(e) {
		const { player, block, dimension } = e;
		const item = player.getComponent("equippable").getEquipment(EquipmentSlot.Mainhand);
		if (item?.hasTag("mc:is_knife")) {
			dimension.spawnItem(new ItemStack(cakeItemMap[block.typeId] || `${block.typeId}_item`, 1), block.bottomCenter());
			removePiecesOrBlock(block, dimension);
			dimension.playSound("block.itemframe.remove_item", block.center());
			itemApplyDamage(player, item);
		}
	}
	onPlayerDestroy(e) {
		const blockId = e.destroyedBlockPermutation.type.id;
		spawnItemSilkTouch(e, cakeItemMap[blockId] || `${blockId}_item`, 4 - e.destroyedBlockPermutation.getState("drop:pieces"));
	}
}
class DropDyeableCake {
	onPlayerInteract(e) {
		const { player, block, dimension } = e;
		const inventory = player.getComponent("inventory").container;
		const item = inventory.getItem(player.selectedSlotIndex);

		if (item?.typeId.includes("minecraft:") && item?.typeId.endsWith("_dye")) {
			const color = item.typeId.replace("minecraft:", "").replace("_dye", "");
			const blockColorMatch = block.typeId.match(/cake_(.*?)_(base|chocolate|gold)/);
			const blockColor = blockColorMatch ? blockColorMatch[1] : null;
			const isBaseBlock = block.typeId === "drop:cake_base" || block.typeId === "drop:cake_chocolate" || block.typeId === "drop:cake_gold";
			if (isBaseBlock || (blockColor && blockColor !== color)) {
				const direction = block.permutation.getState("minecraft:cardinal_direction");
				const pieces = block.permutation.getState("drop:pieces");
				const newBlockType = `drop:cake_${color}_${block.typeId.split("_").pop()}`;
				const newPermutation = BlockPermutation.resolve(newBlockType, {"minecraft:cardinal_direction": direction,"drop:pieces": pieces});
				block.setPermutation(newPermutation);
				dimension.playSound("block.itemframe.add_item", block.location);
				decrementItemInHand(player);
			}
		}
	}
}
system.afterEvents.scriptEventReceive.subscribe(e => {
	const { id, sourceEntity } = e;
	if (id !== "drop:bowl_convert_cheese") return;
	if (sourceEntity?.typeId !== "drop:bowl") return;
	const block = sourceEntity.dimension.getBlock(sourceEntity.location);
	if (block?.typeId === "drop:bowl_0_drop") {
		block.setPermutation(block.permutation.withState("drop:item", "cheese"));
	}
});
class DropBowl {
	onPlayerInteract(e) {
		const { player, block, dimension } = e;
		const item = player.getComponent("inventory").container.getItem(player.selectedSlotIndex);
		const state = block.permutation.getState("drop:item");
		if (state === "empty" && item?.typeId === "minecraft:milk_bucket") {
			block.setPermutation(block.permutation.withState("drop:item", "milk"));
			dimension.playSound("mob.mooshroom.suspicious_milk", block.location);
			addOrReplaceItem(player, "minecraft:bucket", 1);
		} else if (state === "milk" && item?.typeId === "minecraft:leather") {
			dimension.playSound("block.itemframe.add_item", block.location);
			player.onScreenDisplay.setActionBar({rawtext: [{ translate: 'drop.bowl.cheese_placed'}, { translate: 'drop.message.wait'}]});
			dimension.spawnEntity("drop:bowl", {x: block.x + 0.5, y: block.y, z: block.z + 0.5});
			decrementItemInHand(player);
		} else if (state === "cheese") {
			block.setPermutation(block.permutation.withState("drop:item", "empty"));
			dimension.playSound("block.itemframe.remove_item", block.location);
			dimension.getEntitiesAtBlockLocation(block.location).find(entity => entity.typeId === "drop:bowl")?.remove();
			addItemOrSpawn(player, new ItemStack("drop:cheese_block_item", 1));
		}
	}
}
const FOOD_TYPE = {
	"minecraft:chicken": "minecraft:cooked_chicken",
	"minecraft:porkchop": "minecraft:cooked_porkchop",
	"minecraft:beef": "minecraft:cooked_beef",
	"minecraft:mutton": "minecraft:cooked_mutton",
	"minecraft:rabbit": "minecraft:cooked_rabbit",
	"minecraft:cod": "minecraft:cooked_cod",
	"minecraft:salmon": "minecraft:cooked_salmon",
	"minecraft:kelp": "minecraft:dried_kelp",
	"minecraft:potato": "minecraft:baked_potato",
	"minecraft:egg": "drop:fried_egg",
	"drop:raw_fries": "drop:fries",
	"mr:flesh": "mr:cooked_flesh",
	// Add more items here if needed
};
world.afterEvents.dataDrivenEntityTrigger.subscribe(({ entity, eventId }) => {
	if (!entity) return;
	try {
		const rawFood = entity?.getDynamicProperty("drop:raw_food");
		const cookedFood = entity?.getDynamicProperty("drop:cooked_food");
		if (!rawFood || !cookedFood) return entity.remove();
		const { dimension, location } = entity;
		switch (eventId) {
			case "drop:final_cooked":
				entity.runCommand(`replaceitem entity @s slot.weapon.mainhand 0 ${cookedFood}`);
				break;
			case "drop:despawn":
				dimension.spawnItem(new ItemStack(entity.getProperty("drop:cooked") ? cookedFood : rawFood, 1), location);
				entity.remove();
				break;
		}
	} catch (error) { }
}, {
	entityTypes: [ "drop:food_entity" ],
	eventTypes: [ "drop:final_cooked", "drop:despawn" ]
});
class DropFryingPan {
	constructor(){
		this.SKILLET_OFFSETS = [
			{ x: -0.22, z: -0.22 },
			{ x: -0.22, z: 0.22 },
			{ x: 0.22, z: -0.22 },
			{ x: 0.22, z: 0.22 }
		];
		this.onPlayerInteract = this.onPlayerInteract.bind(this);
	}
	getArea(faceLocation, blockLocation) {
		const relX = faceLocation.x - Math.floor(blockLocation.x);
		const relZ = faceLocation.z - Math.floor(blockLocation.z);
		return (relX >= 0.5) << 1 | (relZ >= 0.5);
	}
	getFoodEntityAtArea(entities, area) {
		return entities.find(e => e.getProperty("drop:area") === area);
	}
	manageSkilletEntity(player, dimension, item, spawnPos, area, direction) {
		const entities = dimension.getEntitiesAtBlockLocation(spawnPos);
		const hasSkilletEntity = this.getFoodEntityAtArea(entities, area);
		if (hasSkilletEntity) return;
		const cookedFood = FOOD_TYPE[item.typeId];
		const food = dimension.spawnEntity("drop:food_entity", spawnPos);
		food.setDynamicProperty("drop:raw_food", item.typeId);
		food.setDynamicProperty("drop:cooked_food", cookedFood);
		food.setProperty("drop:area", area);
		food.setRotation({ x: 0, y: { north: 0, south: 180, west: -90, east: 90 }[direction] });
		dimension.playSound("block.itemframe.add_item", spawnPos);
		decrementItemInHand(player);
		food.runCommand(`replaceitem entity @s slot.weapon.mainhand 0 ${item.typeId}`);
	}
	onPlayerInteract(e) {
		const { player, block, dimension, faceLocation } = e;
		const area = this.getArea(faceLocation, block.location);
		const blockCenter = block.bottomCenter();
		const offset = this.SKILLET_OFFSETS[area];
		const spawnPos = {
			x: blockCenter.x + offset.x,
			y: blockCenter.y,
			z: blockCenter.z + offset.z
		};
		const entities = dimension.getEntitiesAtBlockLocation(spawnPos);
		const food = this.getFoodEntityAtArea(entities, area);
		const item = player.getComponent("inventory")?.container?.getItem(player.selectedSlotIndex);
		if (food?.getProperty("drop:cooked")) {
			dimension.playSound("block.itemframe.remove_item", block.location);
			food.triggerEvent("drop:despawn");
			return;
		}
		if (!food && item && FOOD_TYPE[item.typeId]) {
			const direction = block.permutation.getState(directionState);
			this.manageSkilletEntity(player, dimension, item, spawnPos, area, direction);
			return;
		}
		if (!block.permutation.getState("drop:animation") && block.below().matches("drop:rustic_oven")) {
			const allFoodEntities = dimension.getEntitiesAtBlockLocation(block.location).filter(e => e.typeId === "drop:food_entity");
			if (allFoodEntities.length <= 0) return;
			const direction = block.permutation.getState(directionState);
			spawnEntityRotatedByBlock("drop:skillet", block, direction, { init: block.bottomCenter() }).triggerEvent(block.typeId);
			allFoodEntities.forEach(f => f.triggerEvent("drop:flip"));
			block.setPermutation(block.permutation.withState("drop:animation", true));
			system.runTimeout(() => { try {
				block.setPermutation(block.permutation.withState("drop:animation", false));
			} catch (error) {} }, 15.6);
		}
	}
}
const sounds = [
	{ id: "record.13", texture: "textures/items/record_13" },
	{ id: "record.cat", texture: "textures/items/record_cat" },
	{ id: "record.blocks", texture: "textures/items/record_blocks" },
	{ id: "record.chirp", texture: "textures/items/record_chirp" },
	{ id: "record.far", texture: "textures/items/record_far" },
	{ id: "record.mall", texture: "textures/items/record_mall" },
	{ id: "record.mellohi", texture: "textures/items/record_mellohi" },
	{ id: "record.stal", texture: "textures/items/record_stal" },
	{ id: "record.strad", texture: "textures/items/record_strad" },
	{ id: "record.ward", texture: "textures/items/record_ward" },
	{ id: "record.11", texture: "textures/items/record_11" },
	{ id: "record.wait", texture: "textures/items/record_wait" },
	{ id: "record.otherside", texture: "textures/items/record_otherside" },
	{ id: "record.5", texture: "textures/items/record_5" },
	{ id: "record.pigstep", texture: "textures/items/record_pigstep" },
	{ id: "record.relic", texture: "textures/items/music_disc_relic" },
	{ id: "record.creator", texture: "textures/items/music_disc_creator" },
	{ id: "record.creator_music_box", texture: "textures/items/music_disc_creator_music_box" },
	{ id: "record.precipice", texture: "textures/items/music_disc_precipice" },
];
class DropRadio {
	onPlayerInteract(e) {
		const { player, block, dimension } = e;
		function showRadioUI(player, radioEntity) {
			const radioUI = new ActionFormData().title({ translate: 'drop.radio.sound_list' });
			sounds.forEach(sound => {
				radioUI.button({ rawtext: [{ text: '§l' }, { translate: `item.${sound.id.replace(".", "_")}.desc` }] }, sound.texture);
			});
			radioUI.show(player).then(response => {
				if (response.canceled) return;
				const selectedSound = sounds[response.selection].id;
				const oldSound = radioEntity?.getDynamicProperty("drop:sound");
				if (oldSound) {
					player.runCommand(`stopsound @a[r=20] ${oldSound}`);
				}
				if (radioEntity) {
					radioEntity.setDynamicProperty("drop:sound", selectedSound);
				} else {
					const newRadio = dimension.spawnEntity("drop:radio_entity", block.center());
					newRadio.setDynamicProperty("drop:sound", selectedSound);
				}
				dimension.playSound(selectedSound, block.center());
				player.onScreenDisplay.setActionBar({ translate: `item.${selectedSound}.drop` });
			});
		}
		const radioEntity = dimension.getEntitiesAtBlockLocation(block.location).find(entity => entity.typeId === "drop:radio_entity");
		if (player.isSneaking && radioEntity) {
			const oldSound = radioEntity.getDynamicProperty("drop:sound");
			if (oldSound) player.runCommand(`stopsound @a[r=20] ${oldSound}`);
			dimension.playSound("block.itemframe.remove_item", block.center());
			radioEntity.remove();
		} else {
			showRadioUI(player, radioEntity);
		}
	}
}
system.afterEvents.scriptEventReceive.subscribe(e => {
	const { id, sourceEntity } = e;
	if (id !== "drop:radio_stopsound") return;
	if (sourceEntity?.typeId !== "drop:radio_entity") return;
	const sound = sourceEntity.getDynamicProperty("drop:sound");
	if (!sound) return;
	const block = sourceEntity.dimension.getBlock(sourceEntity.location);
	block.dimension.runCommand(`stopsound @a ${sound}`);
});
world.afterEvents.playerBreakBlock.subscribe((e) => {
	const { player, block, dimension } = e;
	if (player.matches({ gameMode: GameMode.creative })) return;
	const item = player.getComponent("equippable")?.getEquipment(EquipmentSlot.Mainhand);
	const fortune = item?.getComponent(ItemEnchantableComponent.componentId)?.getEnchantment("fortune");
	const fortuneLevel = fortune?.level ?? 0;
	const dropChance = 0.1 + fortuneLevel * 0.03;
	if (Math.random() > dropChance) return;
	const amount = Math.min(1 + Math.floor(Math.random() * (fortuneLevel + 1)), 3);
	dimension.spawnItem(new ItemStack("drop:mate_leaf", amount), block.center());
}, {
	blockTypes: [ "minecraft:acacia_leaves", "minecraft:azalea_leaves", "minecraft:azalea_leaves_flowered", "minecraft:birch_leaves", "minecraft:cherry_leaves", "minecraft:dark_oak_leaves", "minecraft:jungle_leaves", "minecraft:mangrove_leaves", "minecraft:oak_leaves", "minecraft:pale_oak_leaves", "minecraft:spruce_leaves" ]
});