import { BlockInventoryComponent, system, world } from '@minecraft/server';

class Raccoon {
    static IDENTIFIER = ['pa:raccoon', 'pa:white_raccoon', 'pa:dark_raccoon', 'pa:brown_raccoon'];
    static VALID_CONTAINERS = ['minecraft:chest'];
    static COOLDOWN_TIME = 120000; // 2 minutes in milliseconds
    static raccoonCooldowns = {}; // Almacenar cooldowns de cada mapache

    constructor() {
        this.runTest();
    }

    runTest() {
        system.runInterval(() => {
            const raccoons = this.getRaccoonEntities();
            if (raccoons.length > 0) {
                raccoons.forEach(raccoon => {

                    // Verificar si el mapache está en cooldown
                    if (this.isInCooldown(raccoon)) {
                        return; // Si está en cooldown, no hacer nada más
                    }

                    const chestBlock = this.getNearbyChestBlock(raccoon);
                    if (chestBlock) {
                        const distance = this.getDistance(raccoon.location, chestBlock.location);
                        if (distance <= 2) {
                            this.stealRandomItemFromChest(chestBlock, raccoon);
                            this.startCooldown(raccoon); // Iniciar cooldown después de robar
                        }
                    }
                });
            }
        }, 100);
    }

    getRaccoonEntities() {
        const entities = world.getDimension('overworld').getEntities();
        const raccoons = [];
        for (const entity of entities) {
            // Verifica si el tipo de entidad es uno de los mapaches válidos
            if (Raccoon.IDENTIFIER.includes(entity.typeId)) {
                raccoons.push(entity);
            }
        }
        return raccoons;
    }

    getNearbyChestBlock(entity) {
        const dimension = world.getDimension('overworld');
        const chestRange = 10;
        const { x, y, z } = entity.location;

        for (let dx = -chestRange; dx <= chestRange; dx++) {
            for (let dy = -chestRange; dy <= chestRange; dy++) {
                for (let dz = -chestRange; dz <= chestRange; dz++) {
                    const block = dimension.getBlock({ x: x + dx, y: y + dy, z: z + dz });
                    if (block && Raccoon.VALID_CONTAINERS.includes(block.typeId)) {
                        return block;
                    }
                }
            }
        }
        return null;
    }

    getDistance(pos1, pos2) {
        const dx = pos1.x - pos2.x;
        const dy = pos1.y - pos2.y;
        const dz = pos1.z - pos2.z;
        return Math.sqrt(dx * dx + dy * dy + dz * dz);
    }

    stealRandomItemFromChest(chestBlock, raccoon) {
        const inventoryComponent = chestBlock.getComponent(BlockInventoryComponent.componentId);

        if (inventoryComponent) {
            const inventory = inventoryComponent.container;
            const items = [];

            // Recolectar todos los ítems del cofre
            for (let i = 0; i < inventory.size; i++) {
                const item = inventory.getItem(i);
                if (item) items.push({ slot: i, item });
            }

            if (items.length === 0) {
                return;
            }

            // Seleccionar un ítem aleatorio del cofre
            const randomIndex = Math.floor(Math.random() * items.length);
            const selectedItem = items[randomIndex];
            const itemStack = selectedItem.item;
            const slot = selectedItem.slot;

            // Ajustar la cantidad del ítem en el cofre
            const newAmount = itemStack.amount - 1;
            if (newAmount > 0) {
                itemStack.amount = newAmount;
                inventory.setItem(slot, itemStack);
            } else {
                inventory.setItem(slot, null);
            }

            // Dar el ítem al mapache
            try {
                raccoon.runCommand(
                    `replaceitem entity @s slot.weapon.mainhand 0 ${itemStack.typeId} 1`
                );
                this.sendMessage(`Raccoon stole: ${itemStack.typeId}`);
            } catch (error) {
                // Manejo de errores si no se puede dar el ítem
            }
        }
    }

    // Verifica si el mapache está en cooldown
    isInCooldown(raccoon) {
        const currentTime = Date.now();
        const lastActionTime = Raccoon.raccoonCooldowns[raccoon.id];
        if (lastActionTime && (currentTime - lastActionTime) < Raccoon.COOLDOWN_TIME) {
            return true;
        }
        return false;
    }

    // Inicia el cooldown para el mapache
    startCooldown(raccoon) {
        Raccoon.raccoonCooldowns[raccoon.id] = Date.now();
    }

    sendMessage(message) {
        world.getDimension('overworld').runCommand(`say ${message}`);
    }
}

new Raccoon();