import { system, world, ItemStack, BlockPermutation, GameMode, EquipmentSlot, ItemEnchantableComponent, Direction, MolangVariableMap } from "@minecraft/server";
import { ModalFormData, ActionFormData } from "@minecraft/server-ui";
import { spawnEntityRotatedByBlock } from '../util/utils';
import { airBlocks, backBlockLocation } from '../util/globalVariables';

import { getAddonConfig, updateAddonConfig } from './Config';

let coffinConfig = getAddonConfig("SleepingCoffin");

world.beforeEvents.worldInitialize.subscribe((e) => {
	e.blockComponentRegistry.registerCustomComponent("coffin:place_1x1x2_bed", new CoffinLargeBlockPlacer("1x1x2", { id: "drop:sleeping_coffin", offset: { y: -0.1, z: 0.5 } }));
	e.blockComponentRegistry.registerCustomComponent("coffin:sleep_in_bed", new CoffinSleepInBed());
	e.blockComponentRegistry.registerCustomComponent("coffin:break_large_block", new CoffinBreakLargeBlock());

	e.itemComponentRegistry.registerCustomComponent("coffin:start_config", new CoffinGlobalConfig());
});
const directionState = "minecraft:cardinal_direction";

const SLEEP_ANIM = {
	"drop:sleeping_coffin": "animation.coffin.player.sleep"
}
world.afterEvents.dataDrivenEntityTrigger.subscribe(({ entity }) => {
	try {
		if (!entity) return;
		const player = entity?.getComponent("rideable")?.getRiders()?.[0];
		if (!player) return;
		player.playAnimation(SLEEP_ANIM[entity?.typeId], { stopExpression: "!q.is_riding" } );
	} catch (error) {}
}, {
	entityTypes: [ "drop:sleeping_coffin" ],
	eventTypes: [ "drop:set_player_anim" ]
});

const overworld = world.getDimension("overworld");
let sleepIntervalId = null;
let addTimeLoop = null;
let sleepingPlayers = [];
let isFading = false;
let targetTime = null;
let requiredSleepingPlayers = false;

const isNightTime = () => {
	const time = world.getTimeOfDay();
	return time >= 13000 && time < 23000;
};
const stopTimeAcceleration = () => {
	if (addTimeLoop) {
		system.clearRun(addTimeLoop);
		targetTime = null;
		addTimeLoop = null;
		stopSleepInterval();
	}
};
function getMinecraftTime(time) {
	const adjustedTime = (time + 6000) % 24000;
	const hours = Math.floor(adjustedTime / 1000);
	const minutes = Math.floor(((adjustedTime % 1000) / 1000) * 60);
	return `${hours.toString().padStart(2, '0')}:${minutes.toString().padStart(2, '0')}`;
}
const showWakeUpEffects = () => {
	if (isFading) return;
	isFading = true;
    const allPlayers = world.getAllPlayers();
	sleepingPlayers.forEach(player => {
		player.camera.fade({
			fadeTime: { fadeInTime: 2.0, holdTime: 0.0, fadeOutTime: 1.0 },
			fadeColor: { red: 0.0, green: 0.0, blue: 0.0 }
		});
	});
	system.runTimeout(() => {
		if (!requiredSleepingPlayers) return;
		overworld.setWeather("Clear");
		world.setTimeOfDay(targetTime || 0);
		allPlayers.forEach(player => {
			player.onScreenDisplay.setActionBar(getMinecraftTime(targetTime || 0));
		});
		CoffinSetOpenAnim();
		isFading = false;
		targetTime = null;
		stopSleepInterval();
	}, 40);
};
const accelerateTimeCycle = () => {
	if (addTimeLoop) return;
	addTimeLoop = system.runInterval(() => {
		const currentTime = world.getTimeOfDay();
		const newTime = (currentTime + 30) % 24000;
		world.setTimeOfDay(newTime);
		const timeDifference = (targetTime - currentTime + 24000) % 24000;
		sleepingPlayers.forEach(player => {
			player.onScreenDisplay.setActionBar(getMinecraftTime(newTime));
		});
		if (!requiredSleepingPlayers) return stopTimeAcceleration();
		if (timeDifference <= 40) {
			overworld.setWeather("Clear");
			world.setTimeOfDay(targetTime || 0);
			CoffinSetOpenAnim();
			stopTimeAcceleration();
		}
	}, 1);
};
//COFFIN
function CoffinSetOpenAnim() {
	sleepingPlayers.forEach(player => {
		const coffinEntity = player.getComponent("minecraft:riding")?.entityRidingOn;
		if (coffinEntity?.matches({ families: [ "sleeping_coffin" ] })) coffinEntity.setProperty("drop:keep_open", true);
	});
}
////////
function startSleepInterval() {
	if (sleepIntervalId !== null) return;
	sleepIntervalId = system.runInterval(() => {
		const allPlayers = world.getAllPlayers();
		sleepingPlayers = allPlayers.filter(player => 
			player.getComponent("minecraft:riding")?.entityRidingOn?.matches({ families: [ "sleeping_coffin" ] })
		);
		const requiredCount = Math.ceil(world.gameRules.playersSleepingPercentage / 100 * allPlayers.length);
		requiredSleepingPlayers = sleepingPlayers.length >= requiredCount;
		if (sleepingPlayers.length === 0) return stopSleepInterval();
		if (isFading) return;
		if (!requiredSleepingPlayers) {
			sleepingPlayers.forEach(player => {
				player.onScreenDisplay.setActionBar({ translate: "drop.actionbar.sleeping_players", with: [`${sleepingPlayers.length}/${requiredCount}`] });
			});
			return;
		} else if (targetTime !== null && requiredSleepingPlayers) {
			if (coffinConfig.accelerateTime) accelerateTimeCycle();
			else showWakeUpEffects();
		}
	}, 20);
}
function stopSleepInterval() {
	if (sleepIntervalId) {
		system.clearRun(sleepIntervalId);
		sleepIntervalId = null;
		targetTime = null;
		isFading = false;
	}
}
class CoffinSleepInBed {
	constructor() {
		this.onPlayerInteract = this.onPlayerInteract.bind(this);
	}
	sleepPlayer(player, bed) {
		bed.getComponent("rideable").addRider(player);
		if (!targetTime) PlayersSleep(player);
	}
	onPlayerInteract(e) {
		const { player, block, dimension } = e;
		let bed = dimension.getEntitiesAtBlockLocation(block.location).find((entity) => entity.matches({ families: [ "sleeping_coffin" ] }));
		if (coffinConfig.spawnEntities) {
			if (!bed) return;
		} else {
			if (!bed) {
				const blockPart = block.permutation.getState("mc:block_parts");
				const direction = block.permutation.getState(directionState);
				const spawnOffset = blockPart === 1 ? { y: -0.1, z: 0.5 } : { y: -0.1, z: -0.5 };

				bed = spawnEntityRotatedByBlock("drop:sleeping_coffin", block, direction, spawnOffset);
				bed.setProperty("drop:despawneable", true);
			}
		}
		if (checkDimensionExplosion(dimension)) return;
		if (bed.getProperty("drop:has_rider")) return;
		checkSpawnPoint(player, player.getSpawnPoint(), { 
			x: Math.round(bed.location.x),
			y: Math.round(bed.location.y),
			z: Math.round(bed.location.z),
			dimension: player.dimension
		});
		if (coffinConfig.onlySleepAtNight) {
			if (isNightTime()) this.sleepPlayer(player, bed);
			else player.sendMessage({ rawtext: [{ text: '§i' }, { translate: 'tile.bed.noSleep' }] });
		} else this.sleepPlayer(player, bed);
	}
}
function PlayersSleep(player) {
	if (!sleepIntervalId) startSleepInterval();
	if (coffinConfig.onlySleepAtNight) {
		targetTime = 0;
		return;
	}
	const SleepUI = new ActionFormData()
		.title({ translate: 'drop.bed.select_time_of_day' })
		.button({ translate: 'hostOption.time.sunrise' }, "textures/coffin/time/time_sunrise")     // 23000
		.button({ translate: 'hostOption.time.day' }, "textures/coffin/time/time_day")             // 1000
		.button({ translate: 'hostOption.time.noon' }, "textures/coffin/time/time_noon")           // 6000
		.button({ translate: 'hostOption.time.sunset' }, "textures/coffin/time/time_sunset")       // 12000
		.button({ translate: 'hostOption.time.night' }, "textures/coffin/time/time_night")         // 13000
		.button({ translate: 'hostOption.time.midnight' }, "textures/coffin/time/time_midnight");  // 18000
	SleepUI.show(player).then(response => {
		if (response.canceled) return;
		const selectedTime = [23000, 1000, 6000, 12000, 13000, 18000][response.selection];
		targetTime = selectedTime;
	});
}
function checkSpawnPoint(player, playerSpawnpoint, newSpawnpoint) {
	const areSpawnPointsEqual = playerSpawnpoint?.dimension.id === newSpawnpoint.dimension.id && playerSpawnpoint?.x === newSpawnpoint.x && playerSpawnpoint?.y === newSpawnpoint.y && playerSpawnpoint?.z === newSpawnpoint.z;
	if (!areSpawnPointsEqual) {
		player.setSpawnPoint(newSpawnpoint);
		player.sendMessage({ rawtext: [{ text: '§i' }, { translate: 'tile.bed.respawnSet' }] });
	}
}
function checkDimensionExplosion(dimension) {
	if (dimension.id !== "minecraft:overworld") {
		dimension.createExplosion(center, 5, {breaksBlocks: true, causesFire: true});
		return true;
	} return false;
}
const blockPartGetters = new Map([
	["1x1x2", (direction, block) => [
		{ target: block.offset(backBlockLocation[direction]), part: 2 }
	]]
]);
class CoffinLargeBlockPlacer {
	constructor(sizeKey, spawnEntity = undefined) {
		this.blockParts = blockPartGetters.get(sizeKey);
		this.spawnEntity = spawnEntity;
		this.onPlace = this.onPlace.bind(this);
		this.beforeOnPlayerPlace = this.beforeOnPlayerPlace.bind(this);
	}
	setBlockParts(parts, id, direction) {
		parts.forEach(({ target, part }) => {
			const params = { [directionState]: direction, "mc:block_parts": part };
			target.setPermutation(BlockPermutation.resolve(id, params));
		});
	}
	beforeOnPlayerPlace(e) {
		const { block, permutationToPlace } = e;
		const direction = permutationToPlace.getState(directionState);
		const parts = this.blockParts(direction, block);
		if (!parts || !parts.every(({ target }) => airBlocks.includes(target.typeId))) e.cancel = true;
	}
	onPlace(e) {
		const { block } = e;
		const direction = block.permutation.getState(directionState);
		const parts = this.blockParts(direction, block);
		this.setBlockParts(parts, block.typeId, direction);
		block.setPermutation(block.permutation.withState("mc:block_parts", 1));
		if (this.spawnEntity && coffinConfig.spawnEntities) spawnEntityRotatedByBlock(this.spawnEntity.id, block, direction, this.spawnEntity.offset).triggerEvent(block.getTags()[0]);
	}
}
class CoffinBreakLargeBlock {
	onPlayerDestroy(e) {
		if (e.player.matches({ gameMode: GameMode.creative })) return;
		const item = e.player.getComponent("equippable").getEquipment(EquipmentSlot.Mainhand);
		const hasSilkTouch = item?.getComponent(ItemEnchantableComponent.componentId)?.getEnchantment("silk_touch");
		if (item && hasSilkTouch) return;
		e.dimension.spawnItem(new ItemStack(e.destroyedBlockPermutation.type.id, 1), e.block.center());
	}
}
class CoffinGlobalConfig {
	constructor() {
		this.onUse = this.onUse.bind(this);
	}
	onUse(e) {
		this.showMainMenu(e.source);
	}
	showMainMenu(player) {
		new ActionFormData()
		.title({ translate: "coffin.main.title" })
		.body({ translate: "coffin.main.body", with: [ "\n" ] })
		.button({ translate: "coffin.main.button.info" }, "textures/buttons/info")
		.button({ translate: "coffin.main.button.settings" }, "textures/buttons/settings")
		.button({ translate: "coffin.main.button.projects" }, "textures/buttons/projects")
		.show(player).then(r => {
			if (r.canceled) return;
			switch(r.selection) {
				case 0: this.showInfoMenu(player); break;
				case 1: this.showSettingsMenu(player); break;
				case 2: this.showOtherProyectsMenu(player); break;
				default: return;
			}
		});
	}
	showInfoMenu(player) {
		new ActionFormData()
		.title({ translate: "coffin.info.title" })
		.body({ translate: "coffin.info.body", with: [ "\n" ] })
		.button({ translate: "coffin.main_button.back" }, "textures/buttons/back")
		.show(player).then(r => {
			if (r.canceled) return;
			this.showMainMenu(player);
		});
	}
	showSettingsMenu(player) {
		new ModalFormData()
		.title({ translate: 'drop.sleeping_coffin.settings' })
		.toggle({ translate: 'drop.decodrop.accelerate_time_when_sleeping' }, coffinConfig.accelerateTime)
		.toggle({ translate: 'drop.decodrop.sleep_only_at_night' }, coffinConfig.onlySleepAtNight)
		.toggle({ translate: 'drop.decodrop.spawn_entities', with: [ "\n" ] }, coffinConfig.spawnEntities)
		.show(player).then(r => {
			if (r.canceled) return this.showMainMenu(player);
			const [accelerateTime, onlySleepAtNight, spawnEntities] = r.formValues;
			coffinConfig = updateAddonConfig("SleepingCoffin", {
				accelerateTime,
				onlySleepAtNight,
				spawnEntities
			});
		});
	}
	showOtherProyectsMenu(player) {
		new ActionFormData()
		.title({ translate: "coffin.projects.title" })
		.body({ translate: "coffin.projects.body", with: [ "\n" ] })
		.button("DecoDrop (Furniture)", "textures/projects/decodrop")
		.button("Insta Farm", "textures/projects/insta_farm")
		.button("MrCrayFish (Unofficial)", "textures/projects/mcf")
		.button("Turkey Add-On", "textures/projects/turkey")
		.button({ translate: "coffin.main_button.back" }, "textures/buttons/back")
		.show(player).then(r => {
			if (r.canceled) return;
			this.showMainMenu(player);
		});
	}
}