import { world } from '@minecraft/server';
import { spawnEntityRotatedByBlock } from '../util/utils';
import { directionState, leftBlockLocation, rightBlockLocation } from '../util/globalVariables';

export class DropSaloonDoor {
	constructor() {
		this.onStepOn = this.onStepOn.bind(this);
		this.onPlace = this.onPlace.bind(this);
		this.beforeOnPlayerPlace = this.beforeOnPlayerPlace.bind(this);
	}
	getMovementSign(facing, velocity) {
		const axisMap = {
			north: velocity.z,
			south: -velocity.z,
			west: velocity.x,
			east: -velocity.x,
		};
		const speed = axisMap[facing] ?? 0;
		return Math.abs(speed) > 0.01 ? Math.sign(speed) : 0;
	}
	createDoor(block, direction, move, sideState, dimension) {
		let door = dimension.getEntitiesAtBlockLocation(block.location).find(en => en.matches({ type: "drop:saloon_door" }));
		if (!door) door = spawnEntityRotatedByBlock("drop:saloon_door", block, direction, { init: block.bottomCenter() });
		door.triggerEvent("drop:open");
		door.triggerEvent(block.typeId);
		door.setProperty("drop:dir", move);
		door.setProperty("drop:side", sideState);
		block.setPermutation(block.permutation.withState("drop:open", true));
	}
	onStepOn(e) {
		const { entity, block, dimension } = e;
		const direction = block.permutation.getState(directionState);
		const sideState = block.permutation.getState("drop:side");
		const velocity = entity.getVelocity();
		const move = this.getMovementSign(direction, velocity);
		this.createDoor(block, direction, move, sideState, dimension);
		const sideBlock = sideState ? block.offset(leftBlockLocation[direction]) : block.offset(rightBlockLocation[direction]);
		if (sideBlock.hasTag("drop:is_saloon_door")) this.createDoor(sideBlock, sideBlock.permutation.getState(directionState), move, sideBlock.permutation.getState("drop:side"), dimension);
	}
	onPlace(e) {
		e.block.setPermutation(e.block.permutation.withState("drop:open", false))
	}
	beforeOnPlayerPlace(e) {
		const { block, permutationToPlace } = e;
		const direction = permutationToPlace.getState(directionState);
		const isRightState = block.offset(leftBlockLocation[direction]).hasTag("drop:side_saloon_door");
		e.permutationToPlace = permutationToPlace.withState("drop:side", isRightState)
	}
}
export function SaloonDoor() {
	world.afterEvents.dataDrivenEntityTrigger.subscribe(({ entity }) => {
		try {
			const block = entity.dimension.getBlock(entity.location);
			if (block.hasTag("drop:is_saloon_door")) {
				block.setPermutation(block.permutation.withState("drop:open", false));
				entity.runCommand("event entity @s drop:despawn");
			}
		} catch (error) {}
	}, {
		entityTypes: [ "drop:saloon_door" ],
		eventTypes: [ "drop:clear_door" ]
	});
}